/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.troubleshooting.internal.operations;

import static java.util.Collections.emptyMap;
import static java.util.List.of;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.equalTo;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.management.stats.AllStatistics;
import org.mule.runtime.core.api.management.stats.FlowConstructStatistics;
import org.mule.runtime.core.api.management.stats.FlowsSummaryStatistics;
import org.mule.runtime.deployment.model.api.artifact.ArtifactContext;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.module.deployment.api.DeploymentService;

import java.io.IOException;
import java.io.StringWriter;
import java.util.Map;

public class StatisticsOperationTestCase {

  private StatisticsOperation statisticsOperation;

  @BeforeEach
  public void setup() {
    // Create mock DeploymentService for testing
    DeploymentService mockDeploymentService = mock(DeploymentService.class);
    statisticsOperation = new StatisticsOperation(mockDeploymentService);
  }

  @Test
  public void testStatisticsOperationDefinition() {
    assertThat(statisticsOperation.getDefinition().getName(), is("statistics"));
    assertThat(statisticsOperation.getDefinition().getDescription(),
               is("Collects statistics information from all deployed applications"));
  }

  @Test
  public void testStatisticsOperationArgument() {
    var arguments = statisticsOperation.getDefinition().getArgumentDefinitions();
    assertThat(arguments, hasSize(1));

    var applicationArgument = arguments.get(0);
    assertThat(applicationArgument.getName(), is("application"));
    assertThat(applicationArgument.getDescription(), is("Application to collect statistics from"));
    assertThat(applicationArgument.isRequired(), is(false));
  }

  @Test
  public void testStatisticsWithNoApplications() throws IOException {
    // Create mock DeploymentService that returns empty list
    DeploymentService mockDeploymentService = mock(DeploymentService.class);
    when(mockDeploymentService.getApplications()).thenReturn(of());

    StatisticsOperation operation = new StatisticsOperation(mockDeploymentService);
    StringWriter writer = new StringWriter();

    operation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    var expected = "No applications deployed.\n";

    // Normalize line endings to handle differences between Windows (\r\n) and Unix (\n)
    String normalizedResult = result.replaceAll("\\r\\n", "\n");
    assertThat(normalizedResult, is(equalTo(expected)));
  }

  @Test
  public void testStatisticsWithApplications() throws IOException {
    Application mockApp = createMockApplication("TestApp");
    DeploymentService mockDeploymentService = mock(DeploymentService.class);
    when(mockDeploymentService.getApplications()).thenReturn(of(mockApp));

    StatisticsOperation operation = new StatisticsOperation(mockDeploymentService);
    StringWriter writer = new StringWriter();

    operation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    var expected = """
        Statistics for application 'TestApp'
        ------------------------------------

        Events Received:                100
        Events Processed:                95
        Messages Dispatched:            200
        Execution Errors:                 5
        Fatal Errors:                     1
        Connection Errors:                2
        Average Processing Time:        150 ms
        Min Processing Time:             50 ms
        Max Processing Time:            500 ms
        Total Processing Time:        15000 ms
        Private Flows Declared:          10
        Private Flows Active:             8
        Trigger Flows Declared:           5
        Trigger Flows Active:             4
        API Kit Flows Declared:           3
        API Kit Flows Active:             2
        Flows statistics:
          flow-1:
            Events Received:             50
            Events Processed:            48
            Messages Dispatched:        100
            Execution Errors:             2
            Fatal Errors:                 0
            Connection Errors:            1
            Average Processing Time:    100 ms

        """;

    // Normalize line endings to handle differences between Windows (\r\n) and Unix (\n)
    String normalizedResult = result.replaceAll("\\r\\n", "\n");
    assertThat(normalizedResult, is(equalTo(expected)));
  }

  @Test
  public void testStatisticsWithSpecificApplication() throws IOException {
    Application mockApp = createMockApplication("TestApp");
    DeploymentService mockDeploymentService = mock(DeploymentService.class);
    when(mockDeploymentService.findApplication("TestApp")).thenReturn(mockApp);

    StatisticsOperation operation = new StatisticsOperation(mockDeploymentService);
    StringWriter writer = new StringWriter();

    // Test with specific application argument
    Map<String, String> arguments = Map.of("application", "TestApp");
    operation.getCallback().execute(arguments, writer);
    String result = writer.toString();

    var expected = """
        Statistics for application 'TestApp'
        ------------------------------------

        Events Received:                100
        Events Processed:                95
        Messages Dispatched:            200
        Execution Errors:                 5
        Fatal Errors:                     1
        Connection Errors:                2
        Average Processing Time:        150 ms
        Min Processing Time:             50 ms
        Max Processing Time:            500 ms
        Total Processing Time:        15000 ms
        Private Flows Declared:          10
        Private Flows Active:             8
        Trigger Flows Declared:           5
        Trigger Flows Active:             4
        API Kit Flows Declared:           3
        API Kit Flows Active:             2
        Flows statistics:
          flow-1:
            Events Received:             50
            Events Processed:            48
            Messages Dispatched:        100
            Execution Errors:             2
            Fatal Errors:                 0
            Connection Errors:            1
            Average Processing Time:    100 ms

        """;

    // Normalize line endings to handle differences between Windows (\r\n) and Unix (\n)
    String normalizedResult = result.replaceAll("\\r\\n", "\n");
    String normalizedExpected = expected.replaceAll("\\r\\n", "\n");
    assertThat(normalizedResult, is(equalTo(normalizedExpected)));
  }

  @Test
  public void testStatisticsWithNonExistentApplication() throws IOException {
    // Create mock DeploymentService that returns null for non-existent app
    DeploymentService mockDeploymentService = mock(DeploymentService.class);
    when(mockDeploymentService.findApplication("NonExistentApp")).thenReturn(null);

    StatisticsOperation operation = new StatisticsOperation(mockDeploymentService);
    StringWriter writer = new StringWriter();

    // Test with non-existent application argument
    Map<String, String> arguments = Map.of("application", "NonExistentApp");
    operation.getCallback().execute(arguments, writer);
    String result = writer.toString();

    var expected = "Application 'NonExistentApp' not found.\n";

    // Normalize line endings to handle differences between Windows (\r\n) and Unix (\n)
    String normalizedResult = result.replaceAll("\\r\\n", "\n");
    assertThat(normalizedResult, is(equalTo(expected)));
  }

  @Test
  public void testStatisticsWithMultipleApplications() throws IOException {
    Application mockApp1 = createMockApplication("TestApp1");
    Application mockApp2 = createMockApplication("TestApp2");
    DeploymentService mockDeploymentService = mock(DeploymentService.class);
    when(mockDeploymentService.getApplications()).thenReturn(of(mockApp1, mockApp2));

    StatisticsOperation operation = new StatisticsOperation(mockDeploymentService);
    StringWriter writer = new StringWriter();

    operation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    var expected = """
        Statistics for application 'TestApp1'
        -------------------------------------

        Events Received:                100
        Events Processed:                95
        Messages Dispatched:            200
        Execution Errors:                 5
        Fatal Errors:                     1
        Connection Errors:                2
        Average Processing Time:        150 ms
        Min Processing Time:             50 ms
        Max Processing Time:            500 ms
        Total Processing Time:        15000 ms
        Private Flows Declared:          10
        Private Flows Active:             8
        Trigger Flows Declared:           5
        Trigger Flows Active:             4
        API Kit Flows Declared:           3
        API Kit Flows Active:             2
        Flows statistics:
          flow-1:
            Events Received:             50
            Events Processed:            48
            Messages Dispatched:        100
            Execution Errors:             2
            Fatal Errors:                 0
            Connection Errors:            1
            Average Processing Time:    100 ms

        Statistics for application 'TestApp2'
        -------------------------------------

        Events Received:                100
        Events Processed:                95
        Messages Dispatched:            200
        Execution Errors:                 5
        Fatal Errors:                     1
        Connection Errors:                2
        Average Processing Time:        150 ms
        Min Processing Time:             50 ms
        Max Processing Time:            500 ms
        Total Processing Time:        15000 ms
        Private Flows Declared:          10
        Private Flows Active:             8
        Trigger Flows Declared:           5
        Trigger Flows Active:             4
        API Kit Flows Declared:           3
        API Kit Flows Active:             2
        Flows statistics:
          flow-1:
            Events Received:             50
            Events Processed:            48
            Messages Dispatched:        100
            Execution Errors:             2
            Fatal Errors:                 0
            Connection Errors:            1
            Average Processing Time:    100 ms

        """;

    // Normalize line endings to handle differences between Windows (\r\n) and Unix (\n)
    String normalizedResult = result.replaceAll("\\r\\n", "\n");
    String normalizedExpected = expected.replaceAll("\\r\\n", "\n");
    assertThat(normalizedResult, is(equalTo(normalizedExpected)));
  }

  private Application createMockApplication(String appName) {
    Application mockApp = mock(Application.class);
    ArtifactContext mockArtifactContext = mock(ArtifactContext.class);
    MuleContext mockMuleContext = mock(MuleContext.class);
    AllStatistics mockStats = mock(AllStatistics.class);
    FlowConstructStatistics mockAppStats = mock(FlowConstructStatistics.class);
    FlowsSummaryStatistics mockFlowStats = mock(FlowsSummaryStatistics.class);
    FlowConstructStatistics mockServiceStat = mock(FlowConstructStatistics.class);

    // Setup mock behavior
    when(mockApp.getArtifactName()).thenReturn(appName);
    when(mockApp.getArtifactContext()).thenReturn(mockArtifactContext);
    when(mockArtifactContext.getMuleContext()).thenReturn(mockMuleContext);
    when(mockMuleContext.getStatistics()).thenReturn(mockStats);
    when(mockStats.getApplicationStatistics()).thenReturn(mockAppStats);
    when(mockStats.getFlowSummaryStatistics()).thenReturn(mockFlowStats);
    when(mockStats.getServiceStatistics()).thenReturn(of(mockServiceStat));

    // Setup application statistics
    when(mockAppStats.getTotalEventsReceived()).thenReturn(100L);
    when(mockAppStats.getProcessedEvents()).thenReturn(95L);
    when(mockAppStats.getTotalDispatchedMessages()).thenReturn(200L);
    when(mockAppStats.getExecutionErrors()).thenReturn(5L);
    when(mockAppStats.getFatalErrors()).thenReturn(1L);
    when(mockAppStats.getConnectionErrors()).thenReturn(2L);
    when(mockAppStats.getAverageProcessingTime()).thenReturn(150L);
    when(mockAppStats.getMinProcessingTime()).thenReturn(50L);
    when(mockAppStats.getMaxProcessingTime()).thenReturn(500L);
    when(mockAppStats.getTotalProcessingTime()).thenReturn(15000L);

    // Setup flow summary statistics
    when(mockFlowStats.getDeclaredPrivateFlows()).thenReturn(10);
    when(mockFlowStats.getActivePrivateFlows()).thenReturn(8);
    when(mockFlowStats.getDeclaredTriggerFlows()).thenReturn(5);
    when(mockFlowStats.getActiveTriggerFlows()).thenReturn(4);
    when(mockFlowStats.getDeclaredApikitFlows()).thenReturn(3);
    when(mockFlowStats.getActiveApikitFlows()).thenReturn(2);

    // Setup service statistics
    when(mockServiceStat.getName()).thenReturn("flow-1");
    when(mockServiceStat.getTotalEventsReceived()).thenReturn(50L);
    when(mockServiceStat.getProcessedEvents()).thenReturn(48L);
    when(mockServiceStat.getTotalDispatchedMessages()).thenReturn(100L);
    when(mockServiceStat.getExecutionErrors()).thenReturn(2L);
    when(mockServiceStat.getFatalErrors()).thenReturn(0L);
    when(mockServiceStat.getConnectionErrors()).thenReturn(1L);
    when(mockServiceStat.getAverageProcessingTime()).thenReturn(100L);

    return mockApp;
  }
}
