/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.troubleshooting.internal.operations;

import static java.lang.System.lineSeparator;
import static org.apache.commons.lang3.StringUtils.leftPad;

import org.mule.runtime.core.api.management.stats.AllStatistics;
import org.mule.runtime.core.api.management.stats.FlowConstructStatistics;
import org.mule.runtime.core.api.management.stats.FlowsSummaryStatistics;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperation;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperationCallback;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperationDefinition;
import org.mule.runtime.module.troubleshooting.api.ArgumentDefinition;
import org.mule.runtime.module.troubleshooting.internal.DefaultTroubleshootingOperationDefinition;
import org.mule.runtime.module.troubleshooting.internal.DefaultArgumentDefinition;

import java.io.IOException;
import java.io.Writer;
import java.util.List;

/**
 * Operation used to collect statistics information from deployed applications.
 * <p>
 * The name of the operation is "statistics".
 * <p>
 * Arguments:
 * <ul>
 * <li>application (Optional): The application to collect statistics from</li>
 * </ul>
 */
public class StatisticsOperation implements TroubleshootingOperation {

  public static final String STATISTICS_OPERATION_NAME = "statistics";
  public static final String STATISTICS_OPERATION_DESCRIPTION =
      "Collects statistics information from all deployed applications";

  public static final String APPLICATION_ARGUMENT_NAME = "application";
  public static final String APPLICATION_ARGUMENT_DESCRIPTION = "Application to collect statistics from";

  private static final TroubleshootingOperationDefinition definition = createOperationDefinition();

  private final DeploymentService deploymentService;

  public StatisticsOperation(DeploymentService deploymentService) {
    this.deploymentService = deploymentService;
  }

  @Override
  public TroubleshootingOperationDefinition getDefinition() {
    return definition;
  }

  @Override
  public TroubleshootingOperationCallback getCallback() {
    return (arguments, writer) -> {
      final String applicationName = arguments.get(APPLICATION_ARGUMENT_NAME);
      if (applicationName == null) {
        writeStatisticsForAllApplications(writer);
      } else {
        Application application = deploymentService.findApplication(applicationName);
        if (application != null) {
          writeStatisticsFor(application, writer);
        } else {
          writer.write("Application '" + applicationName + "' not found." + lineSeparator());
        }
      }
    };
  }

  private void writeStatisticsForAllApplications(Writer writer) throws IOException {
    List<Application> applications = deploymentService.getApplications();
    if (!applications.isEmpty()) {
      for (Application app : applications) {
        writeStatisticsFor(app, writer);
      }
    } else {
      writer.write("No applications deployed." + lineSeparator());
    }
  }

  private static void writeStatisticsFor(Application app, Writer writer) throws IOException {
    var artifactContext = app.getArtifactContext();
    if (artifactContext != null) {
      AllStatistics appStats = artifactContext.getMuleContext().getStatistics();
      if (appStats != null) {
        final var appTitle = "Statistics for application '" + app.getArtifactName() + "'";
        writer.write(appTitle + lineSeparator());
        writer.write(leftPad("", appTitle.length(), "-") + lineSeparator());
        writer.write(lineSeparator());

        writeApplicationStatistics(appStats, writer);
        writeFlowSummaryStatistics(appStats, writer);
        writeFlowStatistics(appStats, writer);

        writer.write(lineSeparator());
      }
    }
  }

  private static void writeApplicationStatistics(AllStatistics appStats, Writer writer) throws IOException {
    FlowConstructStatistics appAppStats = appStats.getApplicationStatistics();
    if (appAppStats != null) {
      writer.write(String.format(""
          + "Events Received:             %6d%n"
          + "Events Processed:            %6d%n"
          + "Messages Dispatched:         %6d%n"
          + "Execution Errors:            %6d%n"
          + "Fatal Errors:                %6d%n"
          + "Connection Errors:           %6d%n"
          + "Average Processing Time:     %6d ms%n"
          + "Min Processing Time:         %6d ms%n"
          + "Max Processing Time:         %6d ms%n"
          + "Total Processing Time:       %6d ms%n",
                                 appAppStats.getTotalEventsReceived(),
                                 appAppStats.getProcessedEvents(),
                                 appAppStats.getTotalDispatchedMessages(),
                                 appAppStats.getExecutionErrors(),
                                 appAppStats.getFatalErrors(),
                                 appAppStats.getConnectionErrors(),
                                 appAppStats.getAverageProcessingTime(),
                                 appAppStats.getMinProcessingTime(),
                                 appAppStats.getMaxProcessingTime(),
                                 appAppStats.getTotalProcessingTime()));
    }
  }

  private static void writeFlowSummaryStatistics(AllStatistics appStats, Writer writer) throws IOException {
    FlowsSummaryStatistics appFlowStats = appStats.getFlowSummaryStatistics();
    if (appFlowStats != null) {
      writer.write(String.format(""
          + "Private Flows Declared:      %6d%n"
          + "Private Flows Active:        %6d%n"
          + "Trigger Flows Declared:      %6d%n"
          + "Trigger Flows Active:        %6d%n"
          + "API Kit Flows Declared:      %6d%n"
          + "API Kit Flows Active:        %6d%n",
                                 appFlowStats.getDeclaredPrivateFlows(),
                                 appFlowStats.getActivePrivateFlows(),
                                 appFlowStats.getDeclaredTriggerFlows(),
                                 appFlowStats.getActiveTriggerFlows(),
                                 appFlowStats.getDeclaredApikitFlows(),
                                 appFlowStats.getActiveApikitFlows()));
    }
  }

  private static void writeFlowStatistics(AllStatistics appStats, Writer writer) throws IOException {
    writer.write("Flows statistics:" + lineSeparator());
    for (FlowConstructStatistics flowStat : appStats.getServiceStatistics()) {
      writer.write("  " + flowStat.getName() + ":" + lineSeparator());
      writer.write(String.format(""
          + "    Events Received:         %6d%n"
          + "    Events Processed:        %6d%n"
          + "    Messages Dispatched:     %6d%n"
          + "    Execution Errors:        %6d%n"
          + "    Fatal Errors:            %6d%n"
          + "    Connection Errors:       %6d%n"
          + "    Average Processing Time: %6d ms%n",
                                 flowStat.getTotalEventsReceived(),
                                 flowStat.getProcessedEvents(),
                                 flowStat.getTotalDispatchedMessages(),
                                 flowStat.getExecutionErrors(),
                                 flowStat.getFatalErrors(),
                                 flowStat.getConnectionErrors(),
                                 flowStat.getAverageProcessingTime()));
    }
  }

  private static TroubleshootingOperationDefinition createOperationDefinition() {
    return new DefaultTroubleshootingOperationDefinition(STATISTICS_OPERATION_NAME, STATISTICS_OPERATION_DESCRIPTION,
                                                         createApplicationArgumentDefinition());
  }

  private static ArgumentDefinition createApplicationArgumentDefinition() {
    return new DefaultArgumentDefinition(APPLICATION_ARGUMENT_NAME, APPLICATION_ARGUMENT_DESCRIPTION, false);
  }
}
