/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.troubleshooting.internal.operations;

import static java.lang.String.format;
import static java.lang.System.lineSeparator;
import static java.util.Set.of;

import static org.apache.commons.lang3.StringUtils.leftPad;
import static org.mule.runtime.core.api.alert.MuleAlertingSupport.AlertNames.ALERT_BACKPRESSURE_TRIGGERED;
import static org.mule.runtime.core.api.alert.MuleAlertingSupport.AlertNames.ALERT_MULE_UNKNOWN_ERROR_RAISED;
import static org.mule.runtime.core.api.alert.MuleAlertingSupport.AlertNames.ALERT_NOT_CONSUMED_STREAM_BUSTED;
import static org.mule.runtime.core.api.alert.MuleAlertingSupport.AlertNames.ALERT_REACTOR_DISCARDED_EVENT;
import static org.mule.runtime.core.api.alert.MuleAlertingSupport.AlertNames.ALERT_REACTOR_DROPPED_ERROR;
import static org.mule.runtime.core.api.alert.MuleAlertingSupport.AlertNames.ALERT_XA_RECOVERY_START_ERROR;

import org.mule.runtime.api.alert.TimedDataAggregation;
import org.mule.runtime.core.api.alert.MuleAlertingSupport;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.runtime.module.troubleshooting.api.ArgumentDefinition;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperation;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperationCallback;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperationDefinition;
import org.mule.runtime.module.troubleshooting.internal.DefaultArgumentDefinition;
import org.mule.runtime.module.troubleshooting.internal.DefaultTroubleshootingOperationDefinition;

import java.io.IOException;
import java.io.Writer;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;

/**
 * Operation used to collect a summary of the alerts for predefined time intervals.
 * <p>
 * The name of the operation is "alertFuseboard".
 * <p>
 * Arguments:
 * <ul>
 * <li>application (Optional): The application to collect the alert fuseboard from</li>
 * </ul>
 */
public class AlertFuseboardOperation implements TroubleshootingOperation {

  public static final String ALERT_FUSEBOARD_OPERATION_NAME = "alertFuseboard";
  public static final String ALERT_FUSEBOARD_OPERATION_DESCRIPTION =
      "Collects a summary of the alerts for predefined time intervals";

  public static final String APPLICATION_ARGUMENT_NAME = "application";
  public static final String APPLICATION_ARGUMENT_DESCRIPTION = "Application to collect the event dump from";

  // Alerts that should not show context
  private static final Set<String> ALERTS_WITHOUT_CONTEXT = of(
                                                               "REACTOR_DROPPED_EVENT", // Context available in event dump
                                                               "ASYNC_LOOGER_RINGBUFFER_FULL" // No additional context needed
  );

  private static final TroubleshootingOperationDefinition definition = createOperationDefinition();

  private final DeploymentService deploymentService;

  public AlertFuseboardOperation(DeploymentService deploymentService) {
    this.deploymentService = deploymentService;
  }

  @Override
  public TroubleshootingOperationDefinition getDefinition() {
    return definition;
  }

  @Override
  public TroubleshootingOperationCallback getCallback() {
    return (arguments, writer) -> {
      final String applicationName = arguments.get(APPLICATION_ARGUMENT_NAME);
      if (applicationName == null) {
        writeAlertFuseboardsForAllApplications(writer);
      } else {
        Application application = deploymentService.findApplication(applicationName);
        writeAlertFuseboardEntries(application, writer);
      }
    };
  }

  private static void writeAlertFuseboardFor(Application application, Writer writer)
      throws IOException {
    final var appsTitle = "Alert Fuseboard for application '" + application.getArtifactName() + "'";
    writer.write(appsTitle + lineSeparator());
    writer.write(leftPad("", appsTitle.length(), "-") + lineSeparator());
    writer.write(lineSeparator());

    writeAlertFuseboardEntries(application, writer);
  }

  private void writeAlertFuseboardsForAllApplications(Writer writer) throws IOException {
    for (Application application : deploymentService.getApplications()) {
      writeAlertFuseboardFor(application, writer);
    }
  }

  private static void writeAlertFuseboardEntries(Application application, Writer writer) throws IOException {
    MuleAlertingSupport muleAlertingSupport = application
        .getArtifactContext()
        .getRegistry()
        .lookupByType(MuleAlertingSupport.class)
        .orElseThrow(() -> new IllegalArgumentException(format("Could not get MuleAlertingSupport for application %s.",
                                                               application.getArtifactName())));

    final var alertSummary = muleAlertingSupport.alertsCountAggregation();
    if (alertSummary.isEmpty()) {
      writer.write("  No alerts triggered during the last hour.");
      writer.write(lineSeparator());
      return;
    }

    // Get context aggregation for alerts that support it
    var alertContexts = muleAlertingSupport.getAlertContexts();

    final Optional<Integer> maxAlertNameLength = alertSummary.keySet().stream()
        .map(String::length)
        .max((o1, o2) -> o1 - o2);
    String formatString = "  * %-" + maxAlertNameLength.orElse(30) + "s    %3d / %3d / %4d / %4d%n";

    for (Entry<String, TimedDataAggregation<Integer>> alertData : alertSummary.entrySet()) {
      String alertName = alertData.getKey();
      TimedDataAggregation<Integer> alertCounts = alertData.getValue();

      writer.write(formatString
          .formatted(alertName,
                     alertCounts.forLast1MinInterval(),
                     alertCounts.forLast5MinsInterval(),
                     alertCounts.forLast15MinsInterval(),
                     alertCounts.forLast60MinsInterval()));

      // Add context information if available and appropriate
      if (alertContexts.containsKey(alertName)) {
        Map<String, Integer> contextCounts = alertContexts.get(alertName);
        if (contextCounts != null) {
          writeAlertContext(writer, alertName, contextCounts, alertCounts.forLast60MinsInterval());
        }
      }
    }
    writer.write(lineSeparator());
  }

  private static void writeAlertContext(Writer writer, String alertName,
                                        Map<String, Integer> contextCounts,
                                        int totalAlerts)
      throws IOException {

    if (ALERTS_WITHOUT_CONTEXT.contains(alertName)) {
      return; // Skip context for these alerts
    }

    if (contextCounts.isEmpty()) {
      return;
    }

    String contextDescription = getContextDescription(alertName);
    writer.write("    " + contextDescription + ":" + lineSeparator());

    // Show all unique contexts with their counts
    for (Entry<String, Integer> entry : contextCounts.entrySet()) {
      String context = entry.getKey();
      Integer count = entry.getValue();
      if (count > 1) {
        writer.write("      - " + context + " (" + count + " times)" + lineSeparator());
      } else {
        writer.write("      - " + context + lineSeparator());
      }
    }

    // Show count of alerts without context
    int contextAlerts = contextCounts.values().stream().mapToInt(Integer::intValue).sum();
    int alertsWithoutContext = totalAlerts - contextAlerts;
    if (alertsWithoutContext > 0) {
      writer.write("      - (no context) (" + alertsWithoutContext + " times)" + lineSeparator());
    }
  }

  private static String getContextDescription(String alertName) {
    switch (alertName) {
      case ALERT_BACKPRESSURE_TRIGGERED:
        return "Flow/Component that exceeded capacity";
      case ALERT_REACTOR_DROPPED_ERROR:
        return "Dropped error details";
      case ALERT_MULE_UNKNOWN_ERROR_RAISED:
        return "Error details";
      case ALERT_NOT_CONSUMED_STREAM_BUSTED:
        return "Originating location";
      case ALERT_XA_RECOVERY_START_ERROR:
        return "Resource that failed recovery";
      case ALERT_REACTOR_DISCARDED_EVENT:
        return "Event details";
      default:
        return "Details";
    }
  }



  private static TroubleshootingOperationDefinition createOperationDefinition() {
    return new DefaultTroubleshootingOperationDefinition(ALERT_FUSEBOARD_OPERATION_NAME, ALERT_FUSEBOARD_OPERATION_DESCRIPTION,
                                                         createApplicationArgumentDefinition());
  }

  private static ArgumentDefinition createApplicationArgumentDefinition() {
    return new DefaultArgumentDefinition(APPLICATION_ARGUMENT_NAME, APPLICATION_ARGUMENT_DESCRIPTION, false);
  }
}
