/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.troubleshooting.internal.operations;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonMap;
import static java.util.Optional.of;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.core.api.alert.MuleAlertingSupport.AlertNames.ALERT_BACKPRESSURE_TRIGGERED;

import java.io.IOException;
import java.io.StringWriter;
import java.time.Instant;
import java.util.Map;
import java.util.TreeMap;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mule.runtime.api.alert.TimedDataAggregation;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.core.api.alert.MuleAlertingSupport;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.deployment.model.api.artifact.ArtifactContext;
import org.mule.runtime.module.deployment.api.DeploymentService;

class AlertFuseboardOperationTestCase {

  private DeploymentService deploymentService;
  private AlertFuseboardOperation alertFuseboardOperation;

  @BeforeEach
  public void setup() {
    deploymentService = mock(DeploymentService.class);
    alertFuseboardOperation = new AlertFuseboardOperation(deploymentService);
  }

  private Map<String, TimedDataAggregation<Integer>> mockAlertsSummary() {
    var alerts = new TreeMap<String, TimedDataAggregation<Integer>>();

    alerts.put("ZZ_TOP_ALERT", new TimedDataAggregation<>(now(), 99999, 9999, 999, 99));
    alerts.put("LNRD_SKNRD_ALERT", new TimedDataAggregation<>(now(), 110, 10, 9, 9));
    alerts.put(ALERT_BACKPRESSURE_TRIGGERED, new TimedDataAggregation<>(now(), 1, 2, 3, 4));

    return alerts;
  }

  private Map<String, Map<String, Integer>> mockAlertsContext() {
    var contexts = new TreeMap<String, Map<String, Integer>>();

    // Mock context for backpressure alerts
    var backpressureContexts = new TreeMap<String, Integer>();
    backpressureContexts.put("Flow 'main-flow' exceeded capacity", 1);
    contexts.put(ALERT_BACKPRESSURE_TRIGGERED, backpressureContexts);

    return contexts;
  }

  private static Instant now() {
    return Instant.now();
  }

  @Test
  void whenNoApplicationIsPassedItReturnsAllApplications() throws IOException {
    // Mock applications
    Application app1 = mockApplication("app1", mockAlertsSummary(), mockAlertsContext());
    Application app2 = mockApplication("app2", new TreeMap<>(), new TreeMap<>());

    when(deploymentService.getApplications()).thenReturn(asList(app1, app2));

    final var writer = new StringWriter();
    alertFuseboardOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    var expected = """
        Alert Fuseboard for application 'app1'
        --------------------------------------

          * BACKPRESSURE_TRIGGERED      1 /   2 /    3 /    4
            Flow/Component that exceeded capacity:
              - Flow 'main-flow' exceeded capacity
              - (no context) (3 times)
          * LNRD_SKNRD_ALERT          110 /  10 /    9 /    9
          * ZZ_TOP_ALERT              99999 / 9999 /  999 /   99

        Alert Fuseboard for application 'app2'
        --------------------------------------

          No alerts triggered during the last hour.
        """;
    assertThat(result, is(equalTo(expected)));
  }

  @Test
  void whenApplicationIsPassedItReturnsOnlyThePassedOne() throws IOException {
    // Mock application
    Application app1 = mockApplication("app1", mockAlertsSummary(), mockAlertsContext());
    when(deploymentService.findApplication("app1")).thenReturn(app1);

    final var writer = new StringWriter();
    alertFuseboardOperation.getCallback().execute(singletonMap("application", "app1"), writer);
    String result = writer.toString();

    var expected = """
          * BACKPRESSURE_TRIGGERED      1 /   2 /    3 /    4
            Flow/Component that exceeded capacity:
              - Flow 'main-flow' exceeded capacity
              - (no context) (3 times)
          * LNRD_SKNRD_ALERT          110 /  10 /    9 /    9
          * ZZ_TOP_ALERT              99999 / 9999 /  999 /   99

        """;
    assertThat(result, is(equalTo(expected)));
  }

  @Test
  void whenBackpressureAlertHasMultipleContextsItShowsDeduplicatedContexts() throws IOException {
    // Test deduplication for BACKPRESSURE_TRIGGERED alerts
    var alertsSummary = new TreeMap<String, TimedDataAggregation<Integer>>();
    alertsSummary.put(ALERT_BACKPRESSURE_TRIGGERED, new TimedDataAggregation<>(now(), 1, 2, 3, 4));

    var alertsContext = new TreeMap<String, Map<String, Integer>>();
    // Multiple contexts, some duplicates
    var backpressureContexts = new TreeMap<String, Integer>();
    backpressureContexts.put("Flow 'main-flow' exceeded capacity", 3); // 3 times
    backpressureContexts.put("Flow 'secondary-flow' exceeded capacity", 1); // 1 time
    alertsContext.put(ALERT_BACKPRESSURE_TRIGGERED, backpressureContexts);

    Application app = mockApplication("testApp", alertsSummary, alertsContext);
    when(deploymentService.findApplication("testApp")).thenReturn(app);
    AlertFuseboardOperation operation = new AlertFuseboardOperation(deploymentService);

    final var writer = new StringWriter();
    operation.getCallback().execute(singletonMap("application", "testApp"), writer);
    String result = writer.toString();

    var expected = """
          * BACKPRESSURE_TRIGGERED      1 /   2 /    3 /    4
            Flow/Component that exceeded capacity:
              - Flow 'main-flow' exceeded capacity (3 times)
              - Flow 'secondary-flow' exceeded capacity

        """;
    assertThat(result, is(equalTo(expected)));
  }



  @Test
  void whenAlertHasNoContextItShowsOnlyCounts() throws IOException {
    // Test that alerts without context (like REACTOR_DROPPED_EVENT) don't show context section
    var alertsSummary = new TreeMap<String, TimedDataAggregation<Integer>>();
    alertsSummary.put("REACTOR_DROPPED_EVENT", new TimedDataAggregation<>(now(), 1, 2, 3, 4));

    var alertsContext = new TreeMap<String, Map<String, Integer>>();
    // Even if context is provided, it should be ignored for this alert type
    var noContextContexts = new TreeMap<String, Integer>();
    noContextContexts.put("context1", 1);
    noContextContexts.put("context2", 1);
    alertsContext.put("REACTOR_DROPPED_EVENT", noContextContexts);

    Application app = mockApplication("testApp", alertsSummary, alertsContext);
    when(deploymentService.findApplication("testApp")).thenReturn(app);
    AlertFuseboardOperation operation = new AlertFuseboardOperation(deploymentService);

    final var writer = new StringWriter();
    operation.getCallback().execute(singletonMap("application", "testApp"), writer);
    String result = writer.toString();

    var expected = """
          * REACTOR_DROPPED_EVENT      1 /   2 /    3 /    4

        """;
    assertThat(result, is(equalTo(expected)));
  }

  private static Application mockApplication(String name, Map<String, TimedDataAggregation<Integer>> alertsSummary,
                                             Map<String, Map<String, Integer>> alertsContext) {
    Application app = mock(Application.class);
    when(app.getArtifactName()).thenReturn(name);

    // Mock MuleAlertingSupport
    MuleAlertingSupport muleAlertingSupport = mock(MuleAlertingSupport.class);
    when(muleAlertingSupport.alertsCountAggregation()).thenReturn(alertsSummary);
    when(muleAlertingSupport.getAlertContexts()).thenReturn(alertsContext);

    // Mock Registry and ArtifactContext
    var registry = mock(Registry.class);
    var artifactContext = mock(ArtifactContext.class);
    when(artifactContext.getRegistry()).thenReturn(registry);
    when(registry.lookupByType(MuleAlertingSupport.class)).thenReturn(of(muleAlertingSupport));
    when(app.getArtifactContext()).thenReturn(artifactContext);

    return app;
  }

}
