/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.troubleshooting.internal.operations;

import static java.lang.Long.parseLong;
import static java.lang.Math.log;
import static java.lang.Math.pow;
import static java.lang.Runtime.getRuntime;
import static java.lang.System.currentTimeMillis;
import static java.lang.System.getProperties;
import static java.lang.System.getProperty;
import static java.lang.System.lineSeparator;
import static java.lang.System.nanoTime;
import static java.lang.management.ManagementFactory.getOperatingSystemMXBean;
import static java.lang.management.ManagementFactory.getRuntimeMXBean;
import static org.mule.runtime.api.util.MuleSystemProperties.SYSTEM_PROPERTY_PREFIX;
import static org.mule.runtime.container.api.MuleFoldersUtil.getMuleBaseFolder;
import static org.mule.runtime.container.api.MuleFoldersUtil.getMuleHomeFolder;
import static org.mule.runtime.core.api.config.MuleManifest.getBuildNumber;
import static org.mule.runtime.core.api.config.MuleManifest.getProductName;
import static org.mule.runtime.core.api.config.MuleManifest.getProductVersion;

import static java.lang.String.format;
import static java.util.function.UnaryOperator.identity;
import static java.util.stream.Collectors.toMap;

import static org.apache.commons.lang3.time.DurationFormatUtils.formatDuration;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperation;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperationCallback;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperationDefinition;
import org.mule.runtime.module.troubleshooting.internal.DefaultTroubleshootingOperationDefinition;
import org.slf4j.Logger;

import java.lang.management.OperatingSystemMXBean;
import java.lang.management.RuntimeMXBean;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;

/**
 * Operation used to collect basic environment and metadata information for the current Mule Runtime.
 * <p>
 * The name of the operation is "basicInfo".
 */
public class BasicInfoOperation implements TroubleshootingOperation {

  public static final String BASIC_INFO_OPERATION_NAME = "basicInfo";
  public static final String BASIC_INFO_OPERATION_DESCRIPTION =
      "Collects basic environment and metadata information for the current Mule Runtime";

  private static final TroubleshootingOperationDefinition definition = createOperationDefinition();
  private static final Logger LOGGER = getLogger(BasicInfoOperation.class.getName());

  // Sensitive properties that should be filtered out
  private static final Set<String> SENSITIVE_PROPERTIES = new HashSet<>(Arrays.asList(
                                                                                      "anypoint.platform.client_secret",
                                                                                      "anypoint.platform.proxy_username",
                                                                                      "anypoint.platform.proxy_password",
                                                                                      "anypoint.platform.encryption_key",
                                                                                      "mule.session.sign.cloudHub.secretKey"));

  @Override
  public TroubleshootingOperationDefinition getDefinition() {
    return definition;
  }

  @Override
  public TroubleshootingOperationCallback getCallback() {
    return (arguments, writer) -> {
      writer.write("Mule:" + lineSeparator());
      writer.write(format("  %s %s (build %s)", getProductName(),
                          getProductVersion(),
                          getBuildNumber())
          + lineSeparator());

      writer.write(format("  mule_home: %s", getMuleHomeFolder().getAbsolutePath())
          + lineSeparator());
      writer.write(format("  mule_base: %s", getMuleBaseFolder().getAbsolutePath())
          + lineSeparator());
      writer.write(lineSeparator());

      // get the properties sorted alphabetically
      writer.write("System Properties:" + lineSeparator());
      Map<String, String> allProperties = getProperties().stringPropertyNames().stream()
          .filter(property -> property.startsWith(SYSTEM_PROPERTY_PREFIX) || // Mule properties
              property.startsWith("com.mulesoft.dw") || // DataWeave properties
              property.startsWith("anypoint.platform")) // API Gateway properties
          .filter(property -> !SENSITIVE_PROPERTIES.contains(property)) // Filter out sensitive properties
          .collect(toMap(identity(), System::getProperty, (v1, v2) -> v1, TreeMap::new));
      for (Entry<String, String> entry : allProperties.entrySet()) {
        writer.write(format("  %s: %s", entry.getKey(), entry.getValue()) + lineSeparator());
      }

      writer.write(lineSeparator());

      writer.write("Java:" + lineSeparator());
      writer.write(format("  Version:   %s", getProperty("java.version")) + lineSeparator());
      writer.write(format("  Vendor:    %s", getProperty("java.vendor")) + lineSeparator());
      writer.write(format("  VM name:   %s", getProperty("java.vm.name")) + lineSeparator());
      writer.write(format("  JAVA_HOME: %s", getProperty("java.home")) + lineSeparator());

      writer.write(lineSeparator());
      writer.write("OS:" + lineSeparator());
      writer.write(format("  Name:      %s", getProperty("os.name")) + lineSeparator());
      writer.write(format("  Version:   %s", getProperty("os.version")) + lineSeparator());
      writer.write(format("  Arch:      %s", getProperty("os.arch")) + lineSeparator());

      writer.write(lineSeparator());

      RuntimeMXBean runtimeMxBean = getRuntimeMXBean();
      writer.write(format("Running time: %s", formatDuration(runtimeMxBean.getUptime(), "d'd' HH:mm:ss.SSS")));
      writer.write(lineSeparator());

      writer.write(lineSeparator());
      writer.write("Process Information:" + lineSeparator());
      writer.write(format("  PID: %d", parseLong(runtimeMxBean.getName().split("@")[0])) + lineSeparator());

      writer.write(lineSeparator());
      writer.write("Report Generation:" + lineSeparator());
      writer.write(format("  Report Millis Time: %d", currentTimeMillis()) + lineSeparator());
      writer.write(format("  Report Nano Time: %d", nanoTime()) + lineSeparator());

      writer.write(lineSeparator());
      writer.write("System Resources:" + lineSeparator());

      // Memory information
      Runtime runtime = getRuntime();
      long totalMemory = runtime.totalMemory();
      long freeMemory = runtime.freeMemory();
      long usedMemory = totalMemory - freeMemory;
      long maxMemory = runtime.maxMemory();

      writer.write(format("  memory.used=%s", formatBytes(usedMemory)) + lineSeparator());
      writer.write(format("  memory.free=%s", formatBytes(freeMemory)) + lineSeparator());
      writer.write(format("  memory.total=%s", formatBytes(totalMemory)) + lineSeparator());
      writer.write(format("  memory.max=%s", formatBytes(maxMemory)) + lineSeparator());
      writer.write(format("  memory.used/total=%.2f%%", (double) usedMemory / totalMemory * 100) + lineSeparator());
      writer.write(format("  memory.used/max=%.2f%%", (double) usedMemory / maxMemory * 100) + lineSeparator());

      // CPU information
      writer.write(format("  load.process=%.2f%%", getProcessCpuLoad()) + lineSeparator());
      writer.write(format("  load.system=%.2f%%", getSystemCpuLoad()) + lineSeparator());
      writer.write(format("  load.systemAverage=%.2f%%", getSystemLoadAverage()) + lineSeparator());
    };
  }

  private static TroubleshootingOperationDefinition createOperationDefinition() {
    return new DefaultTroubleshootingOperationDefinition(BASIC_INFO_OPERATION_NAME, BASIC_INFO_OPERATION_DESCRIPTION);
  }

  private static String formatBytes(long bytes) {
    if (bytes < 1024)
      return bytes + "B";
    int exp = (int) (log(bytes) / log(1024));
    String pre = "KMGTPE".charAt(exp - 1) + "";
    return String.format("%.1f%s", bytes / pow(1024, exp), pre);
  }

  private static double getProcessCpuLoad() {
    try {
      OperatingSystemMXBean osBean = getOperatingSystemMXBean();
      if (osBean instanceof com.sun.management.OperatingSystemMXBean) {
        com.sun.management.OperatingSystemMXBean sunOsBean = (com.sun.management.OperatingSystemMXBean) osBean;
        return sunOsBean.getProcessCpuLoad() * 100;
      } else {
        LOGGER.info("Process CPU load not available");
      }
    } catch (Exception e) {
      LOGGER.info("Failed to get process CPU load", e);
    }
    return -1.0;
  }

  private static double getSystemCpuLoad() {
    try {
      OperatingSystemMXBean osBean = getOperatingSystemMXBean();
      if (osBean instanceof com.sun.management.OperatingSystemMXBean) {
        com.sun.management.OperatingSystemMXBean sunOsBean = (com.sun.management.OperatingSystemMXBean) osBean;
        return sunOsBean.getSystemCpuLoad() * 100;
      } else {
        LOGGER.info("System CPU load not available");
      }
    } catch (Exception e) {
      LOGGER.info("Failed to get system CPU load", e);
    }
    return -1.0;
  }

  private static double getSystemLoadAverage() {
    try {
      OperatingSystemMXBean osBean = getOperatingSystemMXBean();
      double loadAverage = osBean.getSystemLoadAverage();
      if (loadAverage >= 0) {
        return loadAverage * 100;
      } else {
        LOGGER.info("System load average not available");
      }
    } catch (Exception e) {
      LOGGER.info("Failed to get system load average", e);
    }
    return -1.0;
  }

}
