/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.troubleshooting.internal.operations;

import static org.mule.runtime.module.troubleshooting.internal.TroubleshootingTestUtils.mockApplication;
import static org.mule.runtime.module.troubleshooting.internal.TroubleshootingTestUtils.mockDeploymentService;
import static org.mule.runtime.module.troubleshooting.internal.TroubleshootingTestUtils.mockFlowStackEntry;
import static org.mule.runtime.module.troubleshooting.internal.operations.EventDumpOperation.APPLICATION_ARGUMENT_DESCRIPTION;
import static org.mule.runtime.module.troubleshooting.internal.operations.EventDumpOperation.APPLICATION_ARGUMENT_NAME;
import static org.mule.runtime.module.troubleshooting.internal.operations.EventDumpOperation.EVENT_DUMP_OPERATION_DESCRIPTION;
import static org.mule.runtime.module.troubleshooting.internal.operations.EventDumpOperation.EVENT_DUMP_OPERATION_NAME;

import static java.lang.System.lineSeparator;
import static java.time.Clock.fixed;
import static java.time.Instant.now;
import static java.time.Instant.ofEpochMilli;
import static java.time.ZoneId.of;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.Collections.singletonMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsIterableWithSize.iterableWithSize;

import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.core.api.context.notification.FlowCallStack;
import org.mule.runtime.core.api.context.notification.FlowStackElement;
import org.mule.runtime.core.api.event.EventContextService;
import org.mule.runtime.core.api.event.EventContextService.FlowStackEntry;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.deployment.model.api.artifact.ArtifactContext;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.runtime.module.troubleshooting.api.ArgumentDefinition;

import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;

import java.io.IOException;
import java.io.StringWriter;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.junit.Before;
import org.junit.Test;
import org.mockito.MockedStatic;

public class EventDumpOperationTestCase {

  private DeploymentService deploymentService;
  private EventDumpOperation eventDumpOperation;

  @Before
  public void setup() {
    FlowStackEntry flowStackEntry = mockFlowStackEntry("001");
    Application app1 = mockApplication("app1",
                                       // shufled to ensure sorting
                                       mockFlowStackEntry("001_z", flowStackEntry),
                                       // an envent context withput hierarchy
                                       mockFlowStackEntry("abc"),
                                       mockFlowStackEntry("001_1", flowStackEntry),
                                       flowStackEntry);
    Application app2 = mockApplication("app2");
    deploymentService = mockDeploymentService(app1, app2);
    eventDumpOperation = new EventDumpOperation(deploymentService);
  }

  @Test
  public void definitionHasCorrectNameDescriptionAndNumberOfArguments() {
    assertThat(eventDumpOperation.getDefinition().getName(), is(EVENT_DUMP_OPERATION_NAME));
    assertThat(eventDumpOperation.getDefinition().getDescription(), is(EVENT_DUMP_OPERATION_DESCRIPTION));
    assertThat(eventDumpOperation.getDefinition().getArgumentDefinitions(), iterableWithSize(1));
  }

  @Test
  public void applicationArgumentDefinitionIsCorrect() {
    ArgumentDefinition applicationArgumentDefinition = eventDumpOperation.getDefinition().getArgumentDefinitions().get(0);
    assertThat(applicationArgumentDefinition.getName(), is(APPLICATION_ARGUMENT_NAME));
    assertThat(applicationArgumentDefinition.getDescription(), is(APPLICATION_ARGUMENT_DESCRIPTION));
    assertThat(applicationArgumentDefinition.isRequired(), is(false));
  }

  @Test
  public void whenNoApplicationIsPassedItReturnsAllApplications() throws IOException {
    final StringWriter writer = new StringWriter();
    executeEventDump(emptyMap(), writer);
    String result = writer.toString();

    String expected =
        "Active Events for application 'app1'" + lineSeparator()
            + "------------------------------------" + lineSeparator()
            + "" + lineSeparator()
            + "Total Event Contexts:      4" + lineSeparator()
            + "Total Root Contexts:       2" + lineSeparator()
            + "" + lineSeparator()
            + "" + lineSeparator()
            + "\"001\" hierarchy" + lineSeparator()
            + "" + lineSeparator()
            + "    \"001_1\", running for: 00:00.000, state: EXECUTING" + lineSeparator()
            + "        at ns:component@MockLocation(null) 66 ms" + lineSeparator()
            + "" + lineSeparator()
            + "    \"001_z\", running for: 00:00.000, state: EXECUTING" + lineSeparator()
            + "        at ns:component@MockLocation(null) 66 ms" + lineSeparator()
            + "" + lineSeparator()
            + "\"001\", running for: 00:00.000, state: EXECUTING" + lineSeparator()
            + "    at ns:component@MockLocation(null) 66 ms" + lineSeparator()
            + "" + lineSeparator()
            + "\"abc\", running for: 00:00.000, state: EXECUTING" + lineSeparator()
            + "    at ns:component@MockLocation(null) 66 ms" + lineSeparator()
            + "" + lineSeparator()
            + "Active Events for application 'app2'" + lineSeparator()
            + "------------------------------------" + lineSeparator()
            + "" + lineSeparator()
            + "Total Event Contexts:      0" + lineSeparator()
            + "Total Root Contexts:       0" + lineSeparator()
            + lineSeparator();
    assertThat(result, is(equalTo(expected)));
  }

  @Test
  public void whenApplicationIsPassedItReturnsOnlyThePassedOne() throws IOException {
    Map<String, String> argumentsWithApplication = new HashMap<>();
    argumentsWithApplication.put(APPLICATION_ARGUMENT_NAME, "app1");
    final StringWriter writer = new StringWriter();
    executeEventDump(argumentsWithApplication, writer);
    String result = writer.toString();

    String expected =
        "Total Event Contexts:      4" + lineSeparator()
            + "Total Root Contexts:       2" + lineSeparator()
            + "" + lineSeparator()
            + "" + lineSeparator()
            + "\"001\" hierarchy" + lineSeparator()
            + "" + lineSeparator()
            + "    \"001_1\", running for: 00:00.000, state: EXECUTING" + lineSeparator()
            + "        at ns:component@MockLocation(null) 66 ms" + lineSeparator()
            + "" + lineSeparator()
            + "    \"001_z\", running for: 00:00.000, state: EXECUTING" + lineSeparator()
            + "        at ns:component@MockLocation(null) 66 ms" + lineSeparator()
            + "" + lineSeparator()
            + "\"001\", running for: 00:00.000, state: EXECUTING" + lineSeparator()
            + "    at ns:component@MockLocation(null) 66 ms" + lineSeparator()
            + "" + lineSeparator()
            + "\"abc\", running for: 00:00.000, state: EXECUTING" + lineSeparator()
            + "    at ns:component@MockLocation(null) 66 ms" + lineSeparator()
            + lineSeparator();
    assertThat(result, is(equalTo(expected)));
  }

  private void executeEventDump(Map<String, String> args, final StringWriter writer) throws IOException {
    Instant instant = now(fixed(ofEpochMilli(66), of("UTC")));
    try (MockedStatic<Instant> mockedStatic = mockStatic(Instant.class)) {
      mockedStatic.when(Instant::now).thenReturn(instant);

      eventDumpOperation.getCallback().execute(args, writer);
    }
  }

  @Test(expected = IllegalArgumentException.class)
  public void whenTheEventContextServiceIsNotPresentItRaisesAnException() throws IOException {
    for (Application application : deploymentService.getApplications()) {
      Registry registry = application.getArtifactContext().getRegistry();
      when(registry.lookupByName(EventContextService.REGISTRY_KEY)).thenReturn(empty());
    }

    Map<String, String> arguments = new HashMap<>();
    arguments.put(APPLICATION_ARGUMENT_NAME, "app1");
    final StringWriter writer = new StringWriter();
    executeEventDump(arguments, writer);
  }

  @Test
  public void whenChildStackHasParentOnlyUniqueElementsAreShown() throws IOException {
    // Create parent and child FlowStackEntry
    FlowStackEntry parentEntry = mockFlowStackEntry("parent_event");
    FlowStackEntry childEntry = mockFlowStackEntry("child_event", parentEntry);

    // Get the FlowStackElement from parent's stack (it has "MockLocation")
    FlowStackElement parentElement = parentEntry.getFlowCallStack().getElements().get(0);

    // Create child element with location "child" (reusing same ComponentIdentifier pattern as parent)
    final TypedComponentIdentifier tci = TypedComponentIdentifier.builder()
        .identifier(buildFromStringRepresentation("ns:component"))
        .type(OPERATION)
        .build();
    ComponentLocation childLocation = mock(ComponentLocation.class);
    when(childLocation.getComponentIdentifier()).thenReturn(tci);
    when(childLocation.getLocation()).thenReturn("child");

    // Create child element (need to mock Instant.now() for constructor)
    FlowStackElement childElement;
    Instant fixedInstant = ofEpochMilli(0);
    try (MockedStatic<Instant> mockedStatic = mockStatic(Instant.class)) {
      mockedStatic.when(Instant::now).thenReturn(fixedInstant);
      childElement = new FlowStackElement("ChildFlow", "child", childLocation, emptyMap());
    }

    // Replace parent stack (using the element from mockFlowStackEntry, which has "MockLocation")
    FlowCallStack parentStack = mock(FlowCallStack.class);
    when(parentStack.getElements()).thenReturn(singletonList(parentElement));
    when(parentEntry.getFlowCallStack()).thenReturn(parentStack);

    // Replace child stack with child element + parent element (child stacks clone parent and push new)
    FlowCallStack childStack = mock(FlowCallStack.class);
    when(childStack.getElements()).thenReturn(asList(childElement, parentElement));
    when(childEntry.getFlowCallStack()).thenReturn(childStack);

    // Create application with parent and child events
    Application appWithHierarchy = mockApplication("appWithHierarchy", parentEntry, childEntry);

    // Mock EventContextService to return both entries
    EventContextService eventContextService = mock(EventContextService.class);
    when(eventContextService.getCurrentlyActiveFlowStacks()).thenReturn(asList(parentEntry, childEntry));

    Registry registry = mock(Registry.class);
    when(registry.lookupByName(EventContextService.REGISTRY_KEY)).thenReturn(of(eventContextService));

    ArtifactContext artifactContext = mock(ArtifactContext.class);
    when(artifactContext.getRegistry()).thenReturn(registry);
    when(appWithHierarchy.getArtifactContext()).thenReturn(artifactContext);

    deploymentService = mockDeploymentService(appWithHierarchy);
    eventDumpOperation = new EventDumpOperation(deploymentService);

    final StringWriter writer = new StringWriter();
    executeEventDump(singletonMap(APPLICATION_ARGUMENT_NAME, "appWithHierarchy"), writer);
    String result = writer.toString();

    String expected =
        "Total Event Contexts:      2" + lineSeparator()
            + "Total Root Contexts:       1" + lineSeparator()
            + lineSeparator()
            + lineSeparator()
            + "\"parent_event\" hierarchy" + lineSeparator()
            + lineSeparator()
            + "    \"child_event\", running for: 00:00.000, state: EXECUTING" + lineSeparator()
            + "        at ns:component@child(null) 66 ms" + lineSeparator()
            + lineSeparator()
            + "\"parent_event\", running for: 00:00.000, state: EXECUTING" + lineSeparator()
            + "    at ns:component@MockLocation(null) 66 ms" + lineSeparator()
            + lineSeparator();

    // Normalize line endings to handle differences between Windows (\r\n) and Unix (\n)
    String normalizedResult = result.replaceAll("\\r\\n", "\n");
    String normalizedExpected = expected.replaceAll("\\r\\n", "\n");
    assertThat(normalizedResult, is(equalTo(normalizedExpected)));
  }
}
