/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.troubleshooting.internal.operations;

import static java.lang.System.lineSeparator;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.Collections.singletonMap;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.equalTo;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.junit.Before;
import org.junit.Test;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.management.stats.AllStatistics;
import org.mule.runtime.core.api.management.stats.FlowConstructStatistics;
import org.mule.runtime.core.api.management.stats.FlowsSummaryStatistics;
import org.mule.runtime.deployment.model.api.artifact.ArtifactContext;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.runtime.module.troubleshooting.api.ArgumentDefinition;

import java.io.IOException;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class StatisticsOperationTestCase {

  private StatisticsOperation statisticsOperation;

  @Before
  public void setup() {
    // Create mock DeploymentService for testing
    DeploymentService mockDeploymentService = mock(DeploymentService.class);
    statisticsOperation = new StatisticsOperation(mockDeploymentService);
  }

  @Test
  public void testStatisticsOperationDefinition() {
    assertThat(statisticsOperation.getDefinition().getName(), is("statistics"));
    assertThat(statisticsOperation.getDefinition().getDescription(),
               is("Collects statistics information from all deployed applications"));
  }

  @Test
  public void testStatisticsOperationArgument() {
    List<ArgumentDefinition> arguments = statisticsOperation.getDefinition().getArgumentDefinitions();
    assertThat(arguments, hasSize(1));

    ArgumentDefinition applicationArgument = arguments.get(0);
    assertThat(applicationArgument.getName(), is("application"));
    assertThat(applicationArgument.getDescription(), is("Application to collect statistics from"));
    assertThat(applicationArgument.isRequired(), is(false));
  }

  @Test
  public void testStatisticsWithNoApplications() throws IOException {
    // Create mock DeploymentService that returns empty list
    DeploymentService mockDeploymentService = mock(DeploymentService.class);
    when(mockDeploymentService.getApplications()).thenReturn(emptyList());

    StatisticsOperation operation = new StatisticsOperation(mockDeploymentService);
    StringWriter writer = new StringWriter();

    operation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    String expected = "No applications deployed.\n";

    // Normalize line endings to handle differences between Windows (\r\n) and Unix (\n)
    String normalizedResult = result.replaceAll("\\r\\n", "\n");
    assertThat(normalizedResult, is(equalTo(expected)));
  }

  @Test
  public void testStatisticsWithApplications() throws IOException {
    String expected =
        "Statistics for application 'TestApp'" + lineSeparator() +
            "------------------------------------" + lineSeparator() +
            lineSeparator() +
            "Private Flows Declared:          10" + lineSeparator() +
            "Private Flows Active:             8" + lineSeparator() +
            "Trigger Flows Declared:           5" + lineSeparator() +
            "Trigger Flows Active:             4" + lineSeparator() +
            "API Kit Flows Declared:           3" + lineSeparator() +
            "API Kit Flows Active:             2" + lineSeparator() +
            "Events Received:                100" + lineSeparator() +
            "Events Processed:                95" + lineSeparator() +
            "Messages Dispatched:            200" + lineSeparator() +
            "Execution Errors:                 5" + lineSeparator() +
            "Fatal Errors:                     1" + lineSeparator() +
            "Connection Errors:                2" + lineSeparator() +
            "Average Processing Time:        150 ms" + lineSeparator() +
            "Min Processing Time:             50 ms" + lineSeparator() +
            "Max Processing Time:            500 ms" + lineSeparator() +
            "Total Processing Time:        15000 ms" + lineSeparator() +
            "Flows statistics:" + lineSeparator() +
            "  flow-1:" + lineSeparator() +
            "    Events Received:             50" + lineSeparator() +
            "    Events Processed:            48" + lineSeparator() +
            "    Messages Dispatched:        100" + lineSeparator() +
            "    Execution Errors:             2" + lineSeparator() +
            "    Fatal Errors:                 0" + lineSeparator() +
            "    Connection Errors:            1" + lineSeparator() +
            "    Average Processing Time:    100 ms" + lineSeparator() + lineSeparator();

    assertStatisticsWithApplications(expected, true);
  }

  @Test
  public void testStatisticsWithApplicationsAndAppStatisticsDisabled() throws IOException {
    String expected =
        "Statistics for application 'TestApp'" + lineSeparator() +
            "------------------------------------" + lineSeparator() +
            lineSeparator() +
            "Private Flows Declared:          10" + lineSeparator() +
            "Private Flows Active:             8" + lineSeparator() +
            "Trigger Flows Declared:           5" + lineSeparator() +
            "Trigger Flows Active:             4" + lineSeparator() +
            "API Kit Flows Declared:           3" + lineSeparator() +
            "API Kit Flows Active:             2" + lineSeparator() + lineSeparator();

    assertStatisticsWithApplications(expected, false);
  }

  private void assertStatisticsWithApplications(String expected, boolean enableStatistics) throws IOException {
    Application mockApp = createMockApplication("TestApp", enableStatistics);
    DeploymentService mockDeploymentService = mock(DeploymentService.class);
    when(mockDeploymentService.getApplications()).thenReturn(singletonList(mockApp));

    StatisticsOperation operation = new StatisticsOperation(mockDeploymentService);
    StringWriter writer = new StringWriter();

    operation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    // Normalize line endings to handle differences between Windows (\r\n) and Unix (\n)
    String normalizedResult = result.replaceAll("\\r\\n", "\n");
    assertThat(normalizedResult, is(equalTo(expected)));
  }

  @Test
  public void testStatisticsWithSpecificApplication() throws IOException {
    Application mockApp = createMockApplication("TestApp");
    DeploymentService mockDeploymentService = mock(DeploymentService.class);
    when(mockDeploymentService.findApplication("TestApp")).thenReturn(mockApp);

    StatisticsOperation operation = new StatisticsOperation(mockDeploymentService);
    StringWriter writer = new StringWriter();

    // Test with specific application argument
    Map<String, String> arguments = singletonMap("application", "TestApp");
    operation.getCallback().execute(arguments, writer);
    String result = writer.toString();

    String expected =
        "Statistics for application 'TestApp'" + lineSeparator() +
            "------------------------------------" + lineSeparator() +
            lineSeparator() +
            "Private Flows Declared:          10" + lineSeparator() +
            "Private Flows Active:             8" + lineSeparator() +
            "Trigger Flows Declared:           5" + lineSeparator() +
            "Trigger Flows Active:             4" + lineSeparator() +
            "API Kit Flows Declared:           3" + lineSeparator() +
            "API Kit Flows Active:             2" + lineSeparator() +
            "Events Received:                100" + lineSeparator() +
            "Events Processed:                95" + lineSeparator() +
            "Messages Dispatched:            200" + lineSeparator() +
            "Execution Errors:                 5" + lineSeparator() +
            "Fatal Errors:                     1" + lineSeparator() +
            "Connection Errors:                2" + lineSeparator() +
            "Average Processing Time:        150 ms" + lineSeparator() +
            "Min Processing Time:             50 ms" + lineSeparator() +
            "Max Processing Time:            500 ms" + lineSeparator() +
            "Total Processing Time:        15000 ms" + lineSeparator() +
            "Flows statistics:" + lineSeparator() +
            "  flow-1:" + lineSeparator() +
            "    Events Received:             50" + lineSeparator() +
            "    Events Processed:            48" + lineSeparator() +
            "    Messages Dispatched:        100" + lineSeparator() +
            "    Execution Errors:             2" + lineSeparator() +
            "    Fatal Errors:                 0" + lineSeparator() +
            "    Connection Errors:            1" + lineSeparator() +
            "    Average Processing Time:    100 ms" + lineSeparator() + lineSeparator();

    // Normalize line endings to handle differences between Windows (\r\n) and Unix (\n)
    String normalizedResult = result.replaceAll("\\r\\n", "\n");
    String normalizedExpected = expected.replaceAll("\\r\\n", "\n");
    assertThat(normalizedResult, is(equalTo(normalizedExpected)));
  }

  @Test
  public void testStatisticsWithNonExistentApplication() throws IOException {
    // Create mock DeploymentService that returns null for non-existent app
    DeploymentService mockDeploymentService = mock(DeploymentService.class);
    when(mockDeploymentService.findApplication("NonExistentApp")).thenReturn(null);

    StatisticsOperation operation = new StatisticsOperation(mockDeploymentService);
    StringWriter writer = new StringWriter();

    // Test with non-existent application argument
    Map<String, String> arguments = new HashMap<>();
    arguments.put("application", "NonExistentApp");
    operation.getCallback().execute(arguments, writer);
    String result = writer.toString();

    String expected = "Application 'NonExistentApp' not found.\n";

    // Normalize line endings to handle differences between Windows (\r\n) and Unix (\n)
    String normalizedResult = result.replaceAll("\\r\\n", "\n");
    assertThat(normalizedResult, is(equalTo(expected)));
  }

  @Test
  public void testStatisticsWithMultipleApplications() throws IOException {
    Application mockApp1 = createMockApplication("TestApp1");
    Application mockApp2 = createMockApplication("TestApp2");
    DeploymentService mockDeploymentService = mock(DeploymentService.class);
    when(mockDeploymentService.getApplications()).thenReturn(asList(mockApp1, mockApp2));


    StatisticsOperation operation = new StatisticsOperation(mockDeploymentService);
    StringWriter writer = new StringWriter();

    operation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    String expected =
        "Statistics for application 'TestApp1'" + lineSeparator() +
            "-------------------------------------" + lineSeparator() +
            lineSeparator() +
            "Private Flows Declared:          10" + lineSeparator() +
            "Private Flows Active:             8" + lineSeparator() +
            "Trigger Flows Declared:           5" + lineSeparator() +
            "Trigger Flows Active:             4" + lineSeparator() +
            "API Kit Flows Declared:           3" + lineSeparator() +
            "API Kit Flows Active:             2" + lineSeparator() +
            "Events Received:                100" + lineSeparator() +
            "Events Processed:                95" + lineSeparator() +
            "Messages Dispatched:            200" + lineSeparator() +
            "Execution Errors:                 5" + lineSeparator() +
            "Fatal Errors:                     1" + lineSeparator() +
            "Connection Errors:                2" + lineSeparator() +
            "Average Processing Time:        150 ms" + lineSeparator() +
            "Min Processing Time:             50 ms" + lineSeparator() +
            "Max Processing Time:            500 ms" + lineSeparator() +
            "Total Processing Time:        15000 ms" + lineSeparator() +
            "Flows statistics:" + lineSeparator() +
            "  flow-1:" + lineSeparator() +
            "    Events Received:             50" + lineSeparator() +
            "    Events Processed:            48" + lineSeparator() +
            "    Messages Dispatched:        100" + lineSeparator() +
            "    Execution Errors:             2" + lineSeparator() +
            "    Fatal Errors:                 0" + lineSeparator() +
            "    Connection Errors:            1" + lineSeparator() +
            "    Average Processing Time:    100 ms" + lineSeparator() +
            lineSeparator() +
            "Statistics for application 'TestApp2'" + lineSeparator() +
            "-------------------------------------" + lineSeparator() +
            lineSeparator() +
            "Private Flows Declared:          10" + lineSeparator() +
            "Private Flows Active:             8" + lineSeparator() +
            "Trigger Flows Declared:           5" + lineSeparator() +
            "Trigger Flows Active:             4" + lineSeparator() +
            "API Kit Flows Declared:           3" + lineSeparator() +
            "API Kit Flows Active:             2" + lineSeparator() +
            "Events Received:                100" + lineSeparator() +
            "Events Processed:                95" + lineSeparator() +
            "Messages Dispatched:            200" + lineSeparator() +
            "Execution Errors:                 5" + lineSeparator() +
            "Fatal Errors:                     1" + lineSeparator() +
            "Connection Errors:                2" + lineSeparator() +
            "Average Processing Time:        150 ms" + lineSeparator() +
            "Min Processing Time:             50 ms" + lineSeparator() +
            "Max Processing Time:            500 ms" + lineSeparator() +
            "Total Processing Time:        15000 ms" + lineSeparator() +
            "Flows statistics:" + lineSeparator() +
            "  flow-1:" + lineSeparator() +
            "    Events Received:             50" + lineSeparator() +
            "    Events Processed:            48" + lineSeparator() +
            "    Messages Dispatched:        100" + lineSeparator() +
            "    Execution Errors:             2" + lineSeparator() +
            "    Fatal Errors:                 0" + lineSeparator() +
            "    Connection Errors:            1" + lineSeparator() +
            "    Average Processing Time:    100 ms" + lineSeparator() + lineSeparator();

    // Normalize line endings to handle differences between Windows (\r\n) and Unix (\n)
    String normalizedResult = result.replaceAll("\\r\\n", "\n");
    String normalizedExpected = expected.replaceAll("\\r\\n", "\n");
    assertThat(normalizedResult, is(equalTo(normalizedExpected)));
  }

  private Application createMockApplication(String appName) {
    return createMockApplication(appName, true);
  }

  private Application createMockApplication(String appName, boolean enableStatistics) {
    Application mockApp = mock(Application.class);
    ArtifactContext mockArtifactContext = mock(ArtifactContext.class);
    MuleContext mockMuleContext = mock(MuleContext.class);
    AllStatistics mockStats = mock(AllStatistics.class);
    FlowConstructStatistics mockAppStats = mock(FlowConstructStatistics.class);
    FlowsSummaryStatistics mockFlowStats = mock(FlowsSummaryStatistics.class);
    FlowConstructStatistics mockServiceStat = mock(FlowConstructStatistics.class);

    // Setup mock behavior
    when(mockApp.getArtifactName()).thenReturn(appName);
    when(mockApp.getArtifactContext()).thenReturn(mockArtifactContext);
    when(mockArtifactContext.getMuleContext()).thenReturn(mockMuleContext);
    when(mockMuleContext.getStatistics()).thenReturn(mockStats);
    when(mockStats.isEnabled()).thenReturn(enableStatistics);
    when(mockStats.getApplicationStatistics()).thenReturn(mockAppStats);
    when(mockStats.getFlowSummaryStatistics()).thenReturn(mockFlowStats);
    when(mockStats.getServiceStatistics()).thenReturn(singletonList(mockServiceStat));

    // Setup application statistics
    when(mockAppStats.isEnabled()).thenReturn(enableStatistics);
    when(mockAppStats.getTotalEventsReceived()).thenReturn(100L);
    when(mockAppStats.getProcessedEvents()).thenReturn(95L);
    when(mockAppStats.getTotalDispatchedMessages()).thenReturn(200L);
    when(mockAppStats.getExecutionErrors()).thenReturn(5L);
    when(mockAppStats.getFatalErrors()).thenReturn(1L);
    when(mockAppStats.getConnectionErrors()).thenReturn(2L);
    when(mockAppStats.getAverageProcessingTime()).thenReturn(150L);
    when(mockAppStats.getMinProcessingTime()).thenReturn(50L);
    when(mockAppStats.getMaxProcessingTime()).thenReturn(500L);
    when(mockAppStats.getTotalProcessingTime()).thenReturn(15000L);

    // Setup flow summary statistics
    when(mockFlowStats.getDeclaredPrivateFlows()).thenReturn(10);
    when(mockFlowStats.getActivePrivateFlows()).thenReturn(8);
    when(mockFlowStats.getDeclaredTriggerFlows()).thenReturn(5);
    when(mockFlowStats.getActiveTriggerFlows()).thenReturn(4);
    when(mockFlowStats.getDeclaredApikitFlows()).thenReturn(3);
    when(mockFlowStats.getActiveApikitFlows()).thenReturn(2);

    // Setup service statistics
    when(mockServiceStat.getName()).thenReturn("flow-1");
    when(mockServiceStat.isEnabled()).thenReturn(enableStatistics);
    when(mockServiceStat.getTotalEventsReceived()).thenReturn(50L);
    when(mockServiceStat.getProcessedEvents()).thenReturn(48L);
    when(mockServiceStat.getTotalDispatchedMessages()).thenReturn(100L);
    when(mockServiceStat.getExecutionErrors()).thenReturn(2L);
    when(mockServiceStat.getFatalErrors()).thenReturn(0L);
    when(mockServiceStat.getConnectionErrors()).thenReturn(1L);
    when(mockServiceStat.getAverageProcessingTime()).thenReturn(100L);

    return mockApp;
  }
}
