/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.troubleshooting.internal.operations;

import static java.lang.String.format;
import static java.lang.System.lineSeparator;

import org.mule.runtime.api.alert.TimedDataAggregation;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.api.scheduler.SchedulerView;
import org.mule.runtime.api.service.ServiceRepository;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperation;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperationCallback;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperationDefinition;
import org.mule.runtime.module.troubleshooting.internal.DefaultTroubleshootingOperationDefinition;

import java.io.IOException;
import java.io.Writer;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Supplier;

public class SchedulersStateOperation implements TroubleshootingOperation {

  public static final String SCHEDULERS_STATE_OPERATION_NAME = "schedulers";
  public static final String SCHEDULERS_STATE_OPERATION_DESCRIPTION = "Shows the state of currently available schedulers";

  private final Supplier<Optional<SchedulerService>> schedulerService;

  public SchedulersStateOperation(ServiceRepository serviceRepository) {
    this.schedulerService = () -> serviceRepository
        .getServices()
        .stream()
        .filter(SchedulerService.class::isInstance)
        .map(SchedulerService.class::cast)
        .findFirst();
  }

  @Override
  public TroubleshootingOperationCallback getCallback() {
    return (arguments, writer) -> {
      final Optional<SchedulerService> foundSchedulerService = schedulerService.get();
      if (!foundSchedulerService.isPresent()) {
        return;
      }

      final Set<SchedulerView> schedulers = new TreeSet<>((s1, s2) -> s1.getName().compareTo(s2.getName()));
      schedulers.addAll(foundSchedulerService.get().getSchedulers());
      for (SchedulerView schedulerView : schedulers) {
        writeScheduler(writer, schedulerView);
        writer.write(lineSeparator());
      }
    };
  }

  private void writeScheduler(Writer writer, SchedulerView schedulerView) throws IOException {
    writer.write(format("\"%s\" (%s)%n",
                        schedulerView.getName(), schedulerView.getThreadType()));

    if (schedulerView.isShutdown() || schedulerView.isTerminated()) {
      writer.write(format("  Shutdown: %b - Terminated: %b%n",
                          schedulerView.isShutdown(), schedulerView.isTerminated()));
    }
    writer.write(format("  Active tasks: %d%n",
                        schedulerView.getActiveTasks()));
    if (schedulerView.getQueuedTasks() > 0) {
      writer.write(format("  Queued tasks: %d%n",
                          schedulerView.getQueuedTasks()));
    }

    TimedDataAggregation<Integer> rejectionsOverTime = schedulerView.getRejectionsOverTime();
    if (rejectionsOverTime.forLast60MinsInterval() > 0) {
      writer.write(format("   Rejections: %3d / %3d / %4d / %4d%n",
                          rejectionsOverTime.forLast1MinInterval(),
                          rejectionsOverTime.forLast5MinsInterval(),
                          rejectionsOverTime.forLast15MinsInterval(),
                          rejectionsOverTime.forLast60MinsInterval()));
    }

    TimedDataAggregation<Integer> throttlesOverTime = schedulerView.getThrottlesOverTime();
    if (throttlesOverTime.forLast60MinsInterval() > 0) {
      writer.write(format("   Throttles:  %3d / %3d / %4d / %4d%n",
                          throttlesOverTime.forLast1MinInterval(),
                          throttlesOverTime.forLast5MinsInterval(),
                          throttlesOverTime.forLast15MinsInterval(),
                          throttlesOverTime.forLast60MinsInterval()));
    }
  }

  @Override
  public TroubleshootingOperationDefinition getDefinition() {
    return new DefaultTroubleshootingOperationDefinition(SCHEDULERS_STATE_OPERATION_NAME, SCHEDULERS_STATE_OPERATION_DESCRIPTION);
  }

}
