/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.troubleshooting.internal.operations;

import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static java.util.Collections.emptyMap;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.IsNot.not;
import static org.hamcrest.core.StringContains.containsString;

import java.io.IOException;
import java.io.StringWriter;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

public class BasicInfoOperationTestCase {

  private BasicInfoOperation basicInfoOperation;

  @Before
  public void setup() {
    basicInfoOperation = new BasicInfoOperation();
  }

  @After
  public void tearDown() {
    clearProperty("mule.sysprop");
    clearProperty("nonmule.sysprop");
    clearProperty("com.mulesoft.dw.property");
    clearProperty("anypoint.platform.property");
    clearProperty("anypoint.platform.client_secret");
    clearProperty("anypoint.platform.proxy_username");
    clearProperty("anypoint.platform.proxy_password");
    clearProperty("anypoint.platform.encryption_key");
    clearProperty("mule.session.sign.cloudHub.secretKey");
  }

  @Test
  public void muleSystemProperties() throws IOException {
    setProperty("mule.sysprop", "someValue");

    final StringWriter writer = new StringWriter();
    basicInfoOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    assertThat(result, containsString("mule.sysprop: someValue"));
  }

  @Test
  public void nonMuleSystemProperties() throws IOException {
    setProperty("nonmule.sysprop", "someValue");

    final StringWriter writer = new StringWriter();
    basicInfoOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    assertThat(result, not(containsString("nonmule")));
  }

  @Test
  public void dataWeaveSystemProperties() throws IOException {
    setProperty("com.mulesoft.dw.property", "dwValue");

    final StringWriter writer = new StringWriter();
    basicInfoOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    assertThat(result, containsString("com.mulesoft.dw.property: dwValue"));
  }

  @Test
  public void apiGatewaySystemProperties() throws IOException {
    setProperty("anypoint.platform.property", "platformValue");

    final StringWriter writer = new StringWriter();
    basicInfoOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    assertThat(result, containsString("anypoint.platform.property: platformValue"));
  }

  @Test
  public void sensitiveApiGatewayPropertiesAreFiltered() throws IOException {
    setProperty("anypoint.platform.client_secret", "secret123");
    setProperty("anypoint.platform.proxy_username", "proxyuser");
    setProperty("anypoint.platform.proxy_password", "proxypass");
    setProperty("anypoint.platform.encryption_key", "encryptionkey");

    final StringWriter writer = new StringWriter();
    basicInfoOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    // Sensitive properties should be filtered out
    assertThat(result, not(containsString("anypoint.platform.client_secret")));
    assertThat(result, not(containsString("anypoint.platform.proxy_username")));
    assertThat(result, not(containsString("anypoint.platform.proxy_password")));
    assertThat(result, not(containsString("anypoint.platform.encryption_key")));
  }

  @Test
  public void sensitiveMulePropertiesAreFiltered() throws IOException {
    setProperty("mule.session.sign.cloudHub.secretKey", "cloudhubsecret");

    final StringWriter writer = new StringWriter();
    basicInfoOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    // Sensitive property should be filtered out
    assertThat(result, not(containsString("mule.session.sign.cloudHub.secretKey")));
  }
}
