/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.troubleshooting.internal.operations;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.collection.IsIterableWithSize.iterableWithSize;
import static org.junit.Assert.assertThrows;

import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.when;

import static java.lang.Thread.State.RUNNABLE;
import static java.lang.Thread.State.WAITING;

import java.io.IOException;
import java.io.StringWriter;
import java.lang.management.ManagementFactory;
import java.lang.management.ThreadInfo;
import java.lang.management.ThreadMXBean;

import org.junit.Before;
import org.junit.Test;
import org.mockito.MockedStatic;

public class ThreadCpuUsageOperationTestCase {

  private ThreadCpuUsageOperation threadCpuUsageOperation;
  private ThreadMXBean mockThreadMXBean;

  @Before
  public void setup() {
    threadCpuUsageOperation = new ThreadCpuUsageOperation();
    mockThreadMXBean = mock(ThreadMXBean.class);
  }

  @Test
  public void definitionHasCorrectNameDescriptionAndNoArguments() {
    assertThat(threadCpuUsageOperation.getDefinition().getName(),
               is(ThreadCpuUsageOperation.THREAD_CPU_USAGE_OPERATION_NAME));
    assertThat(threadCpuUsageOperation.getDefinition().getDescription(),
               is(ThreadCpuUsageOperation.THREAD_CPU_USAGE_OPERATION_DESCRIPTION));
    assertThat(threadCpuUsageOperation.getDefinition().getArgumentDefinitions(),
               iterableWithSize(0));
  }

  @Test
  public void whenCpuTimeNotSupportedReturnsNotSupportedMessage() throws IOException {
    when(mockThreadMXBean.isThreadCpuTimeSupported()).thenReturn(false);

    try (MockedStatic<ManagementFactory> mockedManagementFactory = mockStatic(ManagementFactory.class)) {
      mockedManagementFactory.when(ManagementFactory::getThreadMXBean).thenReturn(mockThreadMXBean);

      final var writer = new StringWriter();
      threadCpuUsageOperation.getCallback().execute(null, writer);

      String result = writer.toString();
      assertThat(result, containsString("Thread CPU time measurement is not supported on this JVM"));
    }
  }

  @Test
  public void whenCpuTimeSupportedButDisabledEnablesIt() throws IOException {
    when(mockThreadMXBean.isThreadCpuTimeSupported()).thenReturn(true);
    when(mockThreadMXBean.isThreadCpuTimeEnabled()).thenReturn(false);
    when(mockThreadMXBean.getAllThreadIds()).thenReturn(new long[] {1, 2});
    when(mockThreadMXBean.getThreadCpuTime(1L)).thenReturn(1000000000L, 1100000000L); // 1s, then 1.1s
    when(mockThreadMXBean.getThreadCpuTime(2L)).thenReturn(2000000000L, 2000000000L); // 2s, then 2s (no change)

    ThreadInfo mockThreadInfo1 = mock(ThreadInfo.class);
    when(mockThreadInfo1.getThreadName()).thenReturn("test-thread-1");
    when(mockThreadInfo1.getThreadState()).thenReturn(RUNNABLE);

    ThreadInfo mockThreadInfo2 = mock(ThreadInfo.class);
    when(mockThreadInfo2.getThreadName()).thenReturn("test-thread-2");
    when(mockThreadInfo2.getThreadState()).thenReturn(WAITING);

    when(mockThreadMXBean.getThreadInfo(new long[] {1, 2}, 0))
        .thenReturn(new ThreadInfo[] {mockThreadInfo1, mockThreadInfo2});

    try (MockedStatic<ManagementFactory> mockedManagementFactory = mockStatic(ManagementFactory.class)) {
      mockedManagementFactory.when(ManagementFactory::getThreadMXBean).thenReturn(mockThreadMXBean);

      final var writer = new StringWriter();
      threadCpuUsageOperation.getCallback().execute(null, writer);

      String result = writer.toString();
      assertThat(result, containsString("test-thread-1"));

      // Verify CPU time values - Thread 1 should have 100ms CPU time (1100000000L - 1000000000L = 100000000L ns = 100ms)
      assertThat(result, containsString("100.00ms"));
      // Thread 2 should have 0ms CPU time (2000000000L - 2000000000L = 0L ns = 0ms) so it would not show up in the output

      // Verify thread states
      assertThat(result, containsString("RUNNABLE"));
    }
  }

  @Test
  public void whenCpuTimeIsMinusOneSkipsThread() throws IOException {
    when(mockThreadMXBean.isThreadCpuTimeSupported()).thenReturn(true);
    when(mockThreadMXBean.isThreadCpuTimeEnabled()).thenReturn(true);
    when(mockThreadMXBean.getAllThreadIds()).thenReturn(new long[] {1, 2});
    when(mockThreadMXBean.getThreadCpuTime(1L)).thenReturn(1000000000L, 1100000000L);
    when(mockThreadMXBean.getThreadCpuTime(2L)).thenReturn(-1L, -1L); // CPU time not available

    ThreadInfo mockThreadInfo1 = mock(ThreadInfo.class);
    when(mockThreadInfo1.getThreadName()).thenReturn("test-thread-1");
    when(mockThreadInfo1.getThreadState()).thenReturn(RUNNABLE);

    ThreadInfo mockThreadInfo2 = mock(ThreadInfo.class);
    when(mockThreadInfo2.getThreadName()).thenReturn("test-thread-2");
    when(mockThreadInfo2.getThreadState()).thenReturn(RUNNABLE);

    when(mockThreadMXBean.getThreadInfo(new long[] {1, 2}, 0))
        .thenReturn(new ThreadInfo[] {mockThreadInfo1, mockThreadInfo2});

    try (MockedStatic<ManagementFactory> mockedManagementFactory = mockStatic(ManagementFactory.class)) {
      mockedManagementFactory.when(ManagementFactory::getThreadMXBean).thenReturn(mockThreadMXBean);

      final var writer = new StringWriter();
      threadCpuUsageOperation.getCallback().execute(null, writer);

      String result = writer.toString();
      assertThat(result, containsString("test-thread-1"));
      // Should not contain test-thread-2 since CPU time was -1
      assertThat(result, is(not(containsString("test-thread-2"))));
    }
  }

  @Test
  public void whenExceptionOccursThrowsRuntimeException() {
    when(mockThreadMXBean.isThreadCpuTimeSupported()).thenReturn(true);
    when(mockThreadMXBean.isThreadCpuTimeEnabled()).thenReturn(true);
    when(mockThreadMXBean.getAllThreadIds()).thenThrow(new RuntimeException("Test exception"));

    try (MockedStatic<ManagementFactory> mockedManagementFactory = mockStatic(ManagementFactory.class)) {
      mockedManagementFactory.when(ManagementFactory::getThreadMXBean).thenReturn(mockThreadMXBean);

      final var writer = new StringWriter();
      RuntimeException exception = assertThrows(RuntimeException.class,
                                                () -> threadCpuUsageOperation.getCallback().execute(null, writer));

      assertThat(exception.getMessage(), containsString("Failed to get thread CPU usage"));
      assertThat(exception.getCause().getMessage(), is("Test exception"));
    }
  }

  @Test
  public void outputContainsCorrectHeaders() throws IOException {
    when(mockThreadMXBean.isThreadCpuTimeSupported()).thenReturn(true);
    when(mockThreadMXBean.isThreadCpuTimeEnabled()).thenReturn(true);
    when(mockThreadMXBean.getAllThreadIds()).thenReturn(new long[] {1});
    when(mockThreadMXBean.getThreadCpuTime(1L)).thenReturn(1000000000L, 1100000000L);

    ThreadInfo mockThreadInfo = mock(ThreadInfo.class);
    when(mockThreadInfo.getThreadName()).thenReturn("test-thread");
    when(mockThreadInfo.getThreadState()).thenReturn(RUNNABLE);

    when(mockThreadMXBean.getThreadInfo(new long[] {1}, 0))
        .thenReturn(new ThreadInfo[] {mockThreadInfo});

    try (MockedStatic<ManagementFactory> mockedManagementFactory = mockStatic(ManagementFactory.class)) {
      mockedManagementFactory.when(ManagementFactory::getThreadMXBean).thenReturn(mockThreadMXBean);

      final var writer = new StringWriter();
      threadCpuUsageOperation.getCallback().execute(null, writer);

      String result = writer.toString();
      assertThat(result, containsString("Thread ID"));
      assertThat(result, containsString("%CPU"));
      assertThat(result, containsString("Thread Name"));
      assertThat(result, containsString("State"));
      assertThat(result, containsString("CPU Time (3s)"));
      assertThat(result, containsString("Total CPU Time"));
    }
  }
}
