/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.api.profiling.threading;

import org.mule.api.annotation.Experimental;

import java.lang.management.ThreadInfo;
import java.lang.management.ThreadMXBean;

/**
 * This interface contains information about the thread, which is collected in specific contexts of an operation.
 */
@Experimental
public interface ThreadSnapshot {

  /**
   * This value is extracted from {@link ThreadInfo#getBlockedTime()}.
   *
   * <p>
   * Returns the approximate accumulated elapsed time (in milliseconds) that the thread associated with this <tt>ThreadInfo</tt>
   * has blocked to enter or reenter a monitor since thread contention monitoring is enabled. I.e. the total accumulated time the
   * thread has been in the {@link java.lang.Thread.State#BLOCKED BLOCKED} state since thread contention monitoring was last
   * enabled. This method returns <tt>-1</tt> if thread contention monitoring is disabled.
   *
   * <p>
   * The Java virtual machine may measure the time with a high resolution timer. This statistic is reset when the thread
   * contention monitoring is reenabled.
   *
   * @return the approximate accumulated elapsed time in milliseconds that a thread entered the <tt>BLOCKED</tt> state;
   *         <tt>-1</tt> if thread contention monitoring is disabled.
   *
   * @throws java.lang.UnsupportedOperationException if the Java virtual machine does not support this operation.
   *
   * @see ThreadMXBean#isThreadContentionMonitoringSupported
   * @see ThreadMXBean#setThreadContentionMonitoringEnabled
   */
  Long getBlockedTime();

  /**
   * This value is extracted from {@link ThreadInfo#getWaitedTime()}.
   *
   * <p>
   * Returns the approximate accumulated elapsed time (in milliseconds) that the thread associated with this <tt>ThreadInfo</tt>
   * has waited for notification since thread contention monitoring is enabled. I.e. the total accumulated time the thread has
   * been in the {@link java.lang.Thread.State#WAITING WAITING} or {@link java.lang.Thread.State#TIMED_WAITING TIMED_WAITING}
   * state since thread contention monitoring is enabled. This method returns <tt>-1</tt> if thread contention monitoring is
   * disabled.
   *
   * <p>
   * The Java virtual machine may measure the time with a high resolution timer. This statistic is reset when the thread
   * contention monitoring is reenabled.
   *
   * @return the approximate accumulated elapsed time in milliseconds that a thread has been in the <tt>WAITING</tt> or
   *         <tt>TIMED_WAITING</tt> state; <tt>-1</tt> if thread contention monitoring is disabled.
   *
   * @throws java.lang.UnsupportedOperationException if the Java virtual machine does not support this operation.
   *
   * @see ThreadMXBean#isThreadContentionMonitoringSupported
   * @see ThreadMXBean#setThreadContentionMonitoringEnabled
   */
  Long getWaitedTime();

  /**
   * This value is extracted from {@link ThreadMXBean#getThreadCpuTime(long)}.
   *
   * <p>
   * Returns the total CPU time for the current thread in nanoseconds. The returned value is of nanoseconds precision but not
   * necessarily nanoseconds accuracy. If the implementation distinguishes between user mode time and system mode time, the
   * returned CPU time is the amount of time that the thread has executed in user mode or system mode.
   *
   * <p>
   * If the thread of the specified ID is not alive or does not exist, this method returns <tt>-1</tt>. If CPU time measurement is
   * disabled, this method returns <tt>-1</tt>. A thread is alive if it has been started and has not yet died.
   *
   * <p>
   * If CPU time measurement is enabled after the thread has started, the Java virtual machine implementation may choose any time
   * up to and including the time that the capability is enabled as the point where CPU time measurement starts.
   *
   * @return the total CPU time for the current thread if the thread of the specified ID exists, the thread is alive, and CPU time
   *         measurement is enabled; <tt>-1</tt> otherwise.
   *
   * @throws IllegalArgumentException                if {@code id <= 0}.
   * @throws java.lang.UnsupportedOperationException if the Java virtual machine does not support CPU time measurement for other
   *                                                 threads.
   *
   * @see ThreadMXBean#getThreadUserTime
   * @see ThreadMXBean#isThreadCpuTimeSupported
   * @see ThreadMXBean#isThreadCpuTimeEnabled
   * @see ThreadMXBean#setThreadCpuTimeEnabled
   */
  Long getCpuTime();
}
