/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.properties.internal.loader;

import static java.util.ServiceLoader.load;
import static java.util.stream.Stream.concat;
import static java.util.stream.StreamSupport.stream;

import org.mule.runtime.properties.api.ConfigurationPropertiesProviderFactory;

import java.util.stream.Stream;

/**
 * Provides a way for the Mule Runtime Container to load the {@link ConfigurationPropertiesProviderFactory} implementations from
 * the MuleContainer itself and from the mule-plugins in the current deployment, according to the TCCL.
 *
 * @since 1.1
 */
public class ConfigurationPropertiesProviderFactoryLoader {

  /**
   * Looks up implementations of {@link ConfigurationPropertiesProviderFactory} from the Mule container and the plugines
   * accessible form the classloader of a deployable artifact.
   *
   * @param deployableArtifactClassLoader the classloader of a deployable artifact from which mule-plugins
   *                                      {@link ConfigurationPropertiesProviderFactory}s will be loaded.
   * @return all available {@link ConfigurationPropertiesProviderFactory} from the container and the extensions within the
   *         deployable artifact.
   */
  public static Stream<ConfigurationPropertiesProviderFactory> loadConfigurationPropertiesProviderFactories(ClassLoader deployableArtifactClassLoader) {
    if (deployableArtifactClassLoader
        .equals(ConfigurationPropertiesProviderFactory.class.getClassLoader())) {
      // Avoid loading the same services twice
      return stream(((Iterable<ConfigurationPropertiesProviderFactory>) () -> load(ConfigurationPropertiesProviderFactory.class,
                                                                                   deployableArtifactClassLoader)
                                                                                       .iterator())
                                                                                           .spliterator(),
                    false);
    }

    // Load from the container...
    final Stream<ConfigurationPropertiesProviderFactory> fromContainerModuleLayer =
        stream(((Iterable<ConfigurationPropertiesProviderFactory>) () -> load(ConfigurationPropertiesProviderFactory.class,
                                                                              ConfigurationPropertiesProviderFactory.class
                                                                                  .getClassLoader())
                                                                                      .iterator())
                                                                                          .spliterator(),
               false);
    // Load from the extensions of the deployable artifact...
    final Stream<ConfigurationPropertiesProviderFactory> fromExtensions =
        stream(((Iterable<ConfigurationPropertiesProviderFactory>) () -> load(ConfigurationPropertiesProviderFactory.class,
                                                                              deployableArtifactClassLoader)
                                                                                  .iterator())
                                                                                      .spliterator(),
               false);

    return concat(fromContainerModuleLayer, fromExtensions);
  }

}
