/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.properties.api;

import static java.lang.String.format;
import static java.util.ServiceLoader.load;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.component.AbstractComponent;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.ServiceLoader;

import org.slf4j.Logger;

/**
 * Artifact attributes configuration. This class represents a single configuration-attributes element from the configuration. It
 * delegates the operations to {@code InitialisableConfigurationPropertiesProvider}, whose implementation is discovered via SPI.
 *
 * @since 1.0
 */
public class DefaultConfigurationPropertiesProvider extends AbstractComponent
    implements ConfigurationPropertiesProvider, Initialisable {

  private static final Logger LOGGER = getLogger(DefaultConfigurationPropertiesProvider.class);
  private final InitialisableConfigurationPropertiesProvider provider;

  // Keeping the following fields to maintain API compatibility.
  protected Map<String, ConfigurationProperty> configurationAttributes = new HashMap<>();
  protected static final String UNKNOWN = "unknown";
  protected final String fileLocation;
  private final ResourceProvider resourceProvider;

  public DefaultConfigurationPropertiesProvider(String fileLocation, String encoding, ResourceProvider resourceProvider) {
    this.fileLocation = fileLocation;
    this.resourceProvider = resourceProvider;
    DefaultConfigurationPropertiesProviderFactory factory = discoverProviderFactory();
    this.provider = factory.createProvider(fileLocation, encoding, resourceProvider, this);
  }

  public DefaultConfigurationPropertiesProvider(String fileLocation, ResourceProvider resourceProvider) {
    this(fileLocation, null, resourceProvider);
  }

  @Override
  public Optional<ConfigurationProperty> provide(String configurationAttributeKey) {
    return (Optional<ConfigurationProperty>) this.provider.provide(configurationAttributeKey);
  }

  @Override
  public String getDescription() {
    return this.provider.getDescription();
  }

  @Override
  public void initialise() throws InitialisationException {
    this.provider.initialise();
    this.configurationAttributes = this.provider.getConfigurationAttributes();
  }

  @Override
  public String toString() {
    return getDescription();
  }

  /**
   * Discovers and returns an instance of {@link DefaultConfigurationPropertiesProviderFactory} using the Java Service Provider
   * Interface (SPI).
   * <p>
   * This method loads available implementations of {@code DefaultConfigurationPropertiesProviderFactory} using
   * {@link ServiceLoader}. If no implementation is found, it throws an {@link IllegalStateException}. If multiple implementations
   * are found, the first one is returned and a debug log message is generated.
   * </p>
   */
  private DefaultConfigurationPropertiesProviderFactory discoverProviderFactory() {
    ServiceLoader<DefaultConfigurationPropertiesProviderFactory> loader =
        load(DefaultConfigurationPropertiesProviderFactory.class,
             DefaultConfigurationPropertiesProviderFactory.class.getClassLoader());

    Iterator<DefaultConfigurationPropertiesProviderFactory> iterator = loader.iterator();

    if (!iterator.hasNext()) {
      throw new IllegalStateException(format("Could not find %s service implementation through SPI",
                                             DefaultConfigurationPropertiesProviderFactory.class.getName()));
    }

    DefaultConfigurationPropertiesProviderFactory factory = iterator.next();

    if (iterator.hasNext()) {
      LOGGER.debug(format("Found more than one %s service implementation through SPI",
                          DefaultConfigurationPropertiesProviderFactory.class.getName()));
    }

    return factory;
  }

  public String createValueHelper(String key, String value) {
    return createValue(key, value);
  }

  // Keeping the following methods to maintain API compatibility.
  protected InputStream getResourceInputStream(String file) throws IOException {
    return resourceProvider.getResourceAsStream(file);
  }

  protected String createValue(String key, String value) {
    return value;
  }
}
