/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.http.api.client.auth;

import static org.mule.runtime.http.api.AllureConstants.HttpFeature.HTTP_SERVICE;
import static org.mule.runtime.http.api.client.auth.HttpAuthenticationType.BASIC;
import static org.mule.runtime.http.api.client.auth.HttpAuthenticationType.DIGEST;
import static org.mule.runtime.http.api.client.auth.HttpAuthenticationType.NTLM;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.core.Is.is;

import static org.junit.jupiter.api.Assertions.assertThrows;

import org.mule.runtime.http.api.client.auth.HttpAuthentication.HttpNtlmAuthentication;

import org.junit.jupiter.api.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(HTTP_SERVICE)
@Story("Authentication Builder")
public class HttpAuthenticationBuilderTestCase {

  @Test
  void builder() {
    HttpAuthentication auth = HttpAuthentication.builder().type(BASIC).username("Helena").password("s33stra").build();

    assertThat(auth.getType(), is(BASIC));
    assertThat(auth.getUsername(), is("Helena"));
    assertThat(auth.getPassword(), is("s33stra"));
    assertThat(auth.isPreemptive(), is(true));
  }

  @Test
  void ntlmBuilder() {
    HttpNtlmAuthentication auth = HttpNtlmAuthentication.builder().username("Allison").password("suburbia").domain("CA").build();

    assertThat(auth.getType(), is(NTLM));
    assertThat(auth.getUsername(), is("Allison"));
    assertThat(auth.getPassword(), is("suburbia"));
    assertThat(auth.getDomain(), is("CA"));
    assertThat(auth.getWorkstation(), is(nullValue()));
  }

  @Test
  void basic() {
    HttpAuthentication auth = HttpAuthentication.basic("Sarah", "l3D4").build();

    assertThat(auth.getType(), is(BASIC));
    assertThat(auth.getUsername(), is("Sarah"));
    assertThat(auth.getPassword(), is("l3D4"));
  }

  @Test
  void digest() {
    HttpAuthentication auth = HttpAuthentication.digest("Cosima", "324B21").preemptive(false).build();

    assertThat(auth.getType(), is(DIGEST));
    assertThat(auth.getUsername(), is("Cosima"));
    assertThat(auth.getPassword(), is("324B21"));
    assertThat(auth.isPreemptive(), is(false));
  }

  @Test
  void ntlm() {
    HttpNtlmAuthentication auth = HttpAuthentication.ntlm("Rachel", "Cold B Digest")
        .domain("DYAD")
        .workstation("MyPC")
        .build();

    assertThat(auth.getType(), is(NTLM));
    assertThat(auth.getUsername(), is("Rachel"));
    assertThat(auth.getPassword(), is("Cold B Digest"));
    assertThat(auth.isPreemptive(), is(true));
    assertThat(auth.getDomain(), is("DYAD"));
    assertThat(auth.getWorkstation(), is("MyPC"));
  }

  @Test
  void failWithoutType() {
    var thrown =
        assertThrows(NullPointerException.class, () -> HttpAuthentication.builder().username("Tony").password("W4T").build());
    assertThat(thrown.getMessage(), is("An authentication type must be declared."));
  }

  @Test
  void failWithoutUsername() {
    var thrown =
        assertThrows(NullPointerException.class, () -> HttpAuthentication.builder().type(DIGEST).password("C4s70r").build());
    assertThat(thrown.getMessage(), is("A username must be provided."));
  }

  @Test
  void failWithoutPassword() {
    var thrown =
        assertThrows(NullPointerException.class, () -> HttpNtlmAuthentication.builder().type(NTLM).username("Krystal").build());
    assertThat(thrown.getMessage(), is("A password must be provided."));
  }

}
