/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.http.api.server;

import static org.mule.runtime.http.api.AllureConstants.HttpFeature.HTTP_SERVICE;
import static org.mule.runtime.http.api.HttpConstants.Method.GET;
import static org.mule.runtime.http.api.HttpConstants.Method.OPTIONS;
import static org.mule.runtime.http.api.HttpConstants.Method.PATCH;
import static org.mule.runtime.http.api.HttpConstants.Method.POST;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;

import static org.junit.jupiter.api.Assertions.assertThrows;

import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.http.api.HttpConstants.Method;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;

import java.util.LinkedList;

import org.junit.jupiter.api.Test;

import io.qameta.allure.Feature;

@Feature(HTTP_SERVICE)
public class MethodRequestMatcherTestCase {

  @Test
  void cannotBeEmpty() {
    var thrown = assertThrows(IllegalArgumentException.class, () -> MethodRequestMatcher.builder().build());
    assertThat(thrown.getMessage(), is("methods attribute should not be empty"));
  }

  @Test
  void cannotBeEmptyCollection() {
    var thrown = assertThrows(IllegalArgumentException.class, () -> MethodRequestMatcher.builder(new LinkedList<>()).build());
    assertThat(thrown.getMessage(), is("methods attribute should not be empty"));
  }

  @Test
  void cannotBeNullCollection() {
    var thrown = assertThrows(NullPointerException.class, () -> MethodRequestMatcher.builder(null).build());
    assertThat(thrown.getMessage(), is("methods attribute should not be null"));
  }

  @Test
  void cannotBeNullString() {
    String none = null;
    var thrown = assertThrows(NullPointerException.class, () -> MethodRequestMatcher.builder().add(none));
    assertThat(thrown.getMessage(), is("method attribute should not be null"));
  }

  @Test
  void cannotBeNullMethod() {
    Method none = null;
    var thrown = assertThrows(NullPointerException.class, () -> MethodRequestMatcher.builder().add(none));
    assertThat(thrown.getMessage(), is("method attribute should not be null"));
  }

  @Test
  void onlyAcceptsOneMethod() {
    final MethodRequestMatcher matcher = MethodRequestMatcher.builder().add(GET).build();

    final var getRequest = mock(HttpRequest.class);
    when(getRequest.getMethod()).thenReturn(GET.name());
    assertThat(matcher.matches(getRequest), is(true));

    final var postRequest = mock(HttpRequest.class);
    when(postRequest.getMethod()).thenReturn(POST.name());
    assertThat(matcher.matches(postRequest), is(false));
  }

  @Test
  void acceptSeveralMethods() {
    final MethodRequestMatcher matcher = MethodRequestMatcher.builder().add(GET).add(POST).add(PATCH).build();

    final var getRequest = mock(HttpRequest.class);
    when(getRequest.getMethod()).thenReturn(GET.name());
    assertThat(matcher.matches(getRequest), is(true));

    final var posrRequest = mock(HttpRequest.class);
    when(posrRequest.getMethod()).thenReturn(POST.name());
    assertThat(matcher.matches(posrRequest), is(true));

    final var patchRequest = mock(HttpRequest.class);
    when(patchRequest.getMethod()).thenReturn(PATCH.name());
    assertThat(matcher.matches(patchRequest), is(true));

    final var optionsRequest = mock(HttpRequest.class);
    when(optionsRequest.getMethod()).thenReturn(OPTIONS.name());
    assertThat(matcher.matches(optionsRequest), is(false));
  }
}
