/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.tracing.level.impl.config;

import static org.mule.runtime.features.api.MuleSystemProperties.ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY;
import static org.mule.runtime.tracing.level.api.config.TracingLevel.DEBUG;
import static org.mule.runtime.tracing.level.api.config.TracingLevel.MONITORING;
import static org.mule.runtime.tracing.level.api.config.TracingLevel.OVERVIEW;
import static org.mule.tck.junit4.rule.SystemProperty.callWithProperty;
import static org.mule.test.allure.AllureConstants.Profiling.PROFILING;
import static org.mule.test.allure.AllureConstants.Profiling.ProfilingServiceStory.TRACING_CONFIGURATION;

import static java.lang.Boolean.getBoolean;
import static java.lang.Thread.currentThread;
import static java.nio.file.Files.copy;
import static java.nio.file.Files.createTempDirectory;
import static java.nio.file.Path.of;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static java.util.Arrays.asList;
import static java.util.concurrent.Executors.callable;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.config.internal.model.dsl.config.PropertyNotFoundException;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.tracing.level.api.config.TracingLevel;
import org.mule.tck.junit4.rule.SystemProperty;

import java.io.IOException;
import java.net.URL;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Properties;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(Parameterized.class)
@Feature(PROFILING)
@Story(TRACING_CONFIGURATION)
public class FileTracingLevelConfigurationTestCase {

  private static final String CONF_FOLDER = "conf";
  private static final String LOCATION_1 = "location1";
  private static final String LOCATION_2 = "location2";
  private static final TracingLevel DEFAULT_LEVEL = MONITORING;

  private static final String TRACING_LEVEL_CONF = "tracing-level.conf";
  private static final String TRACING_LEVEL_EMPTY_CONF = "tracing-level-empty.conf";
  private static final String TRACING_LEVEL_WITH_DUPLICATE_OVERRIDE_CONF = "tracing-level-with-duplicate-override.conf";
  private static final String TRACING_LEVEL_WITH_OVERRIDES_CONF = "tracing-level-with-overrides.conf";
  private static final String TRACING_LEVEL_WITH_SYSTEM_PROPERTY_CONF = "tracing-level-with-system-property.conf";
  private static final String TRACING_LEVEL_WITH_UNRESOLVABLE_SYSTEM_PROPERTY_CONF =
      "tracing-level-with-unresolvable-system-property.conf";
  private static final String TRACING_LEVEL_WITH_WRONG_OVERRIDE_CONF = "tracing-level-with-wrong-override.conf";
  private static final String TRACING_LEVEL_WITH_WRONG_LEVEL_CONF = "tracing-level-with-wrong-level.conf";
  private static final String NON_EXISTENT_FILE_TRACING_LEVEL_CONF = "non-existent-tracing-level.conf";

  @Parameterized.Parameters(name = "enableConfigInApplication: {0}")
  public static Collection<Object[]> data() {
    return asList(new Object[][] {{true}, {false}});
  }

  private MuleContext muleContext;

  @Rule
  public final SystemProperty enableConfigAtApplicationLevelSystemProperty;

  @Before
  public void setUp() {
    muleContext = mock(MuleContext.class);
    when(muleContext.getExecutionClassLoader()).thenReturn(currentThread().getContextClassLoader());
    when(muleContext.getDeploymentProperties()).thenReturn(new Properties());
    when(muleContext.getId()).thenReturn("test-artifact");
  }

  public FileTracingLevelConfigurationTestCase(boolean enableConfigAtApplicationLevel) {
    this.enableConfigAtApplicationLevelSystemProperty =
        new SystemProperty(ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY,
                           Boolean.toString(enableConfigAtApplicationLevel));
  }


  @Test
  public void whenLevelIsSpecifiedInFileItIsReturned() {
    FileTracingLevelConfiguration configuration = createConfiguration(TRACING_LEVEL_CONF);
    assertThat(configuration.getTracingLevel(), is(OVERVIEW));
  }

  @Test
  public void whenNoPropertyIsInTheFileDefaultLevelIsReturned() {
    FileTracingLevelConfiguration configuration = createConfiguration(TRACING_LEVEL_EMPTY_CONF);
    assertThat(configuration.getTracingLevel(), is(DEFAULT_LEVEL));
  }

  @Test
  public void whenNoFileExistsDefaultLevelIsReturned() {
    // ApplicationLevelConfiguration#getSignalConfigurationFileDirectoryPath() throws an exception to prevent fallback to
    // filesystem. Always use TemporaryDirectoryConfiguration to be able to test the fallback behavior:
    // - When (enableConfigAtApplicationLevel == true): tests fallback from classpath to filesystem
    // - When (enableConfigAtApplicationLevel == false): tests direct filesystem access
    // Both cases should return the default level because the required file does not exist.
    FileTracingLevelConfiguration configuration =
        new TemporaryDirectoryConfiguration(muleContext, NON_EXISTENT_FILE_TRACING_LEVEL_CONF);
    assertThat(configuration.getTracingLevel(), is(DEFAULT_LEVEL));
  }

  @Test
  public void whenLevelIsWrongInFileDefaultLevelIsReturned() {
    FileTracingLevelConfiguration configuration = createConfiguration(TRACING_LEVEL_WITH_WRONG_LEVEL_CONF);
    assertThat(configuration.getTracingLevel(), is(DEFAULT_LEVEL));
  }

  @Test
  public void whenALocationOverrideIsSpecifiedInTheFileTheOverrideIsReturned() {
    FileTracingLevelConfiguration configuration =
        createConfiguration(TRACING_LEVEL_WITH_OVERRIDES_CONF);
    assertThat(configuration.getTracingLevelOverride(LOCATION_1), is(MONITORING));
    assertThat(configuration.getTracingLevelOverride(LOCATION_2), is(DEBUG));
  }

  @Test
  public void whenAWrongLocationOverrideIsSpecifiedInTheFileTheDefaultLevelIsReturned() {
    FileTracingLevelConfiguration configuration =
        createConfiguration(TRACING_LEVEL_WITH_WRONG_OVERRIDE_CONF);
    assertThat(configuration.getTracingLevelOverride(LOCATION_1), is(OVERVIEW));
    assertThat(configuration.getTracingLevelOverride(LOCATION_2), is(DEBUG));
  }

  @Test
  public void whenALocationOverrideIsSpecifiedAndDuplicatedInTheFileTheLastOverrideIsReturned() {
    FileTracingLevelConfiguration configuration =
        createConfiguration(TRACING_LEVEL_WITH_DUPLICATE_OVERRIDE_CONF);
    assertThat(configuration.getTracingLevelOverride(LOCATION_1), is(DEBUG));
    assertThat(configuration.getTracingLevelOverride(LOCATION_2), is(DEBUG));
  }

  @Test
  public void whenTracingLevelContainsSystemPropertyItIsResolved() throws Throwable {
    callWithProperty("test.tracing.level", "DEBUG", callable(() -> {
      FileTracingLevelConfiguration configuration =
          createConfiguration(TRACING_LEVEL_WITH_SYSTEM_PROPERTY_CONF);
      assertThat(configuration.getTracingLevel(), is(DEBUG));
    }));
  }

  @Test(expected = PropertyNotFoundException.class)
  public void whenSystemPropertyCannotBeResolvedAnExceptionIsRaised() {
    // Don't set the corresponding property - should throw PropertyNotFoundException
    FileTracingLevelConfiguration configuration =
        createConfiguration(TRACING_LEVEL_WITH_UNRESOLVABLE_SYSTEM_PROPERTY_CONF);
    configuration.getTracingLevel();
  }

  private FileTracingLevelConfiguration createConfiguration(String configurationFileName) {
    FileTracingLevelConfiguration configuration;
    if (getBoolean(ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY)) {
      configuration = new ApplicationLevelConfiguration(muleContext, configurationFileName);
    } else {
      configuration = new TemporaryDirectoryConfiguration(muleContext, configurationFileName);
    }
    return configuration;
  }

  private static class TemporaryDirectoryConfiguration extends FileTracingLevelConfiguration {

    private final String configurationFileName;
    private final Path configurationFileDirectoryPath;

    protected TemporaryDirectoryConfiguration(MuleContext muleContext, String configurationFileName) {
      super(muleContext);
      this.configurationFileName = configurationFileName;
      configurationFileDirectoryPath = createConfigurationFileDirectoryPath(configurationFileName);
    }

    private Path createConfigurationFileDirectoryPath(String confFileName) {
      try {
        // Create a temporary directory to use as the configuration directory
        Path temporaryConfigurationDirectory = createTempDirectory(this.getClass().getName());
        temporaryConfigurationDirectory.toFile().deleteOnExit();

        // Copy the desired configuration file (if existent) into the temporary directory
        URL sourceConfigUrl = getClass().getClassLoader().getResource(CONF_FOLDER + "/" + confFileName);
        if (sourceConfigUrl != null) {
          Path sourceConfigPath = of(sourceConfigUrl.getPath());
          Path destinationConfigPath = temporaryConfigurationDirectory.resolve(confFileName);
          destinationConfigPath.toFile().deleteOnExit();
          copy(sourceConfigPath, destinationConfigPath, REPLACE_EXISTING);
        }

        return temporaryConfigurationDirectory.toAbsolutePath();
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    }

    @Override
    protected String getSignalConfigurationFileName() {
      return configurationFileName;
    }

    @Override
    protected Path getSignalConfigurationFileDirectoryPath() {
      return configurationFileDirectoryPath;
    }
  }

  private static class ApplicationLevelConfiguration extends FileTracingLevelConfiguration {

    private final String configurationResourcePath;

    protected ApplicationLevelConfiguration(MuleContext muleContext, String configurationResourcePath) {
      super(muleContext);
      this.configurationResourcePath = CONF_FOLDER + "/" + configurationResourcePath;
    }

    @Override
    protected String getSignalConfigurationFileName() {
      // This is not a filename but a path to be used to resolve the configuration file from the classpath using getResource
      return configurationResourcePath;
    }

    @Override
    protected Path getSignalConfigurationFileDirectoryPath() {
      throw new AssertionError("This method should not be called on this test when application level configuration is working");
    }
  }
}
