/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.tracing.level.impl.config;

import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.container.api.MuleFoldersUtil.getConfFolder;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_EXPORTER_DEFAULT_TRACING_LEVEL;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_TRACING_LEVEL_CONFIGURATION_FILE_NAME;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_TRACING_LEVEL_CONFIGURATION_FILE_PATH;
import static org.mule.runtime.tracing.level.api.config.TracingLevel.MONITORING;
import static org.mule.runtime.tracing.level.api.config.TracingLevel.valueOf;

import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.nio.file.Paths.get;
import static java.util.Collections.emptyList;
import static java.util.Collections.synchronizedList;
import static java.util.Locale.ROOT;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.module.observability.AbstractFileObservabilitySignalConfiguration;
import org.mule.runtime.module.observability.configuration.ObservabilityConfigurationFileWatcher;
import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfigurationFileFinder;
import org.mule.runtime.tracing.level.api.config.TracingLevel;
import org.mule.runtime.tracing.level.api.config.TracingLevelConfiguration;

import java.io.File;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.Consumer;

import org.slf4j.Logger;

/**
 * A {@link TracingLevelConfiguration} based on a file in the conf folder.
 *
 * @since 4.5.0
 */
public class FileTracingLevelConfiguration extends AbstractFileObservabilitySignalConfiguration
    implements TracingLevelConfiguration, Disposable {

  private static final Logger LOGGER = getLogger(FileTracingLevelConfiguration.class);

  private static final String WRONG_LEVEL_ERROR =
      "Wrong tracing level found in configuration file: {}. The tracing level will be set to the default level: {}";
  private static final String WRONG_LEVEL_OVERRIDE_ERROR =
      "Wrong tracing level override found in configuration file: {}. This override will be ignored.";
  private static final String UNEXPECTED_ERROR_WHILE_READING_FROM_CONFIGURATION_FILE =
      "Unexpected error while reading from configuration file.";
  public static final String CONFIGURATION_FILE_NAME = "tracing-level.conf";

  private static final String LEVEL_PROPERTY_NAME = "mule.openTelemetry.tracer.level";
  private static final String OVERRIDES_PROPERTY_NAME = "mule.openTelemetry.tracer.levelOverrides";


  private final List<Runnable> onConfigurationChangeRunnables = synchronizedList(new ArrayList<>());
  private TracingLevel tracingLevel;
  private Map<String, TracingLevel> tracingLevelOverrides;
  private ObservabilityConfigurationFileWatcher fileWatcher;

  private static ObservabilitySignalConfigurationFileFinder getFileFinder(MuleContext muleContext) {
    return path -> findArtifactConfigFile(muleContext.getExecutionClassLoader(), path);
  }

  public FileTracingLevelConfiguration(MuleContext muleContext) {
    super(getFileFinder(muleContext), muleContext.getDeploymentProperties(), muleContext.getId());
  }

  @Override
  protected void onConfigurationFileNotFound() {
    LOGGER.atInfo()
        .log("Tracing level configuration file named '{}' not found at the '{}' configuration path. Using default configuration.",
             getSignalConfigurationFileName(), getSignalConfigurationFileDirectoryPath());
  }

  @Override
  protected void onConfigurationFileLoadError(Exception error, File configurationFile) {
    // Logging is considered critical feature, therefore we rise the configuration parse error.
    String message = format("Parsing error in the tracing level configuration file '%s'.", configurationFile.getAbsolutePath());
    throw new MuleRuntimeException(createStaticMessage(message, error));
  }

  @Override
  protected String getSignalConfigurationFileName() {
    return getProperty(MULE_OPEN_TELEMETRY_TRACING_LEVEL_CONFIGURATION_FILE_NAME, CONFIGURATION_FILE_NAME);
  }

  @Override
  protected Path getSignalConfigurationFileDirectoryPath() {
    return get(getProperty(MULE_OPEN_TELEMETRY_TRACING_LEVEL_CONFIGURATION_FILE_PATH, getConfFolder().getAbsolutePath()))
        .toAbsolutePath();
  }

  @Override
  protected boolean isArtifactLevelProperty(String configurationKey) {
    return false;
  }

  @Override
  public void initialise() {
    super.initialise();
    configureFileWatcher();
  }

  @Override
  public void dispose() {
    if (fileWatcher != null) {
      fileWatcher.interrupt();
    }
  }

  private TracingLevel computeTracingLevel() {
    TracingLevel computedTracingLevel =
        valueOf(getProperty(MULE_OPEN_TELEMETRY_EXPORTER_DEFAULT_TRACING_LEVEL, MONITORING.toString())
            .toUpperCase());
    try {
      String configuredTracingLevel = getStringValue(LEVEL_PROPERTY_NAME);
      if (configuredTracingLevel != null) {
        try {
          computedTracingLevel = valueOf(configuredTracingLevel.toUpperCase(ROOT));
        } catch (IllegalArgumentException e) {
          LOGGER.error(WRONG_LEVEL_ERROR, configuredTracingLevel, computedTracingLevel);
        }
      }
    } catch (Exception e) {
      LOGGER.error(UNEXPECTED_ERROR_WHILE_READING_FROM_CONFIGURATION_FILE, e);
      throw e;
    }
    return computedTracingLevel;
  }

  private Map<String, TracingLevel> computeTracingLevelsOverrides() {
    Map<String, TracingLevel> computedLevelOverrides = new HashMap<>();
    try {
      getStringListValue(OVERRIDES_PROPERTY_NAME, emptyList()).forEach((override) -> {
        String[] levelOverride = override.split("=");
        if (levelOverride.length != 2) {
          LOGGER.error(WRONG_LEVEL_OVERRIDE_ERROR, override);
        } else {
          try {
            computedLevelOverrides.put(levelOverride[0], valueOf(levelOverride[1].toUpperCase(Locale.ROOT)));
          } catch (IllegalArgumentException e) {
            LOGGER.error(WRONG_LEVEL_OVERRIDE_ERROR, override);
          }
        }
      });
    } catch (Exception e) {
      LOGGER.error(UNEXPECTED_ERROR_WHILE_READING_FROM_CONFIGURATION_FILE, e);
    }
    return computedLevelOverrides;
  }

  private void configureFileWatcher() {
    // TODO W-19296402: Generalize the configuration reloading logic.
    if (getConfigurationFile() != null && getConfigurationFile().exists() && fileWatcher != null) {
      fileWatcher = new ObservabilityConfigurationFileWatcher(
                                                              getConfigurationFile(),
                                                              () -> onConfigurationChangeRunnables.forEach(Runnable::run));
      fileWatcher.start();
    }
  }

  @Override
  public TracingLevel getTracingLevel() {
    if (tracingLevel == null) {
      tracingLevel = computeTracingLevel();
    }
    return tracingLevel;
  }

  @Override
  public TracingLevel getTracingLevelOverride(String location) {
    if (tracingLevelOverrides == null) {
      tracingLevelOverrides = computeTracingLevelsOverrides();
    }
    return tracingLevelOverrides.getOrDefault(location, getTracingLevel());
  }

  @Override
  public void onConfigurationChange(Consumer<TracingLevelConfiguration> onConfigurationChangeConsumer) {
    onConfigurationChangeRunnables.add(() -> onConfigurationChangeConsumer.accept(this));
  }
}
