/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.tracer.exporter.impl.optel.config;

import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.disposeIfNeeded;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfiguration;

import java.util.List;

import org.slf4j.Logger;

/**
 * A composite {@link ObservabilitySignalConfiguration}.
 */
public class CompositeSpanExporterConfiguration implements ObservabilitySignalConfiguration, Disposable {

  private static final Logger LOGGER = getLogger(CompositeSpanExporterConfiguration.class);

  private final List<ObservabilitySignalConfiguration> spanExporterConfigurations;

  public CompositeSpanExporterConfiguration(List<ObservabilitySignalConfiguration> spanExporterConfigurations) {
    this.spanExporterConfigurations = spanExporterConfigurations;
  }

  @Override
  public String getStringValue(String key) {
    return spanExporterConfigurations.stream()
        .filter(configuration -> configuration.getStringValue(key) != null)
        .findFirst()
        .map(configuration -> configuration.getStringValue(key))
        .orElse(null);
  }

  @Override
  public String getStringValue(String key, String defaultValue) {
    return spanExporterConfigurations.stream()
        .filter(configuration -> configuration.getStringValue(key) != null)
        .findFirst()
        .map(configuration -> configuration.getStringValue(key))
        .orElse(defaultValue);
  }

  @Override
  public void doOnConfigurationChanged(Runnable doOnConfigurationChanged) {
    spanExporterConfigurations.forEach(configuration -> configuration.doOnConfigurationChanged(doOnConfigurationChanged));
  }

  @Override
  public void dispose() {
    spanExporterConfigurations.forEach(configuration -> disposeIfNeeded(configuration, LOGGER));
  }
}
