/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.plugin.maven;

import static java.lang.String.format;
import static org.apache.commons.lang.StringUtils.join;
import static org.mule.plugin.maven.PluginProperties.ARTIFACT_EXPORT_CLASS_PACKAGES;
import static org.mule.plugin.maven.PluginProperties.ARTIFACT_EXPORT_RESOURCES;
import static org.mule.plugin.maven.PluginProperties.PLUGIN_BUNDLE;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Properties;

import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;


/**
 * Build a Mule plugin archive.
 */
@Mojo(name = "package", defaultPhase = LifecyclePhase.PACKAGE, requiresDependencyResolution = ResolutionScope.RUNTIME)
public class PluginPackageMojo extends AbstractPackagePluginMojo {

  public static final String MULE_PLUGIN_JSON_LOCATION = RESERVED_FOLDER_MULE_ARTIFACT + MULE_PLUGIN_JSON;
  public static final String META_INF = "META-INF";
  public static final String MULE_ARTIFACT = "mule-artifact";
  public static final String CLASSES = "classes";
  @Parameter(defaultValue = "${project.basedir}", readonly = true)
  private File baseDir;
  /**
   * If provided, it will be out in the final archive. If not, it will be generated based on the extension manifest.
   */
  @Parameter(defaultValue = PLUGIN_PROPERTIES_FILE_NAME)
  private File customPluginProperties;

  @Override
  protected void addToArchiver(ModuleArchiver archiver) throws MojoFailureException {
    addPOMFile(archiver);
    addPluginDescriptor(archiver);
    addClasses(archiver);
  }

  /**
   * Adds the {@link #MULE_PLUGIN_JSON} file, extracted from the current extensions JAR into the {@link #META_INF} folder.
   *
   * @param archiver to store the current {@link #MULE_PLUGIN_JSON} file
   * @throws MojoFailureException
   */
  public void addPluginDescriptor(ModuleArchiver archiver) throws MojoFailureException {
    Path pluginPropertiesPath = Paths.get(outputDirectory.getAbsolutePath(), CLASSES, PLUGIN_PROPERTIES_FILE_NAME);
    if (pluginPropertiesPath.toFile().exists()) {
      doAddArtifactProperties(pluginPropertiesPath.toFile());
      archiver.addFile(pluginPropertiesPath.toFile(), PLUGIN_PROPERTIES_FILE_NAME);
    } else {
      Path pluginJsonPath = Paths.get(outputDirectory.getAbsolutePath(), CLASSES, META_INF, MULE_ARTIFACT, MULE_PLUGIN_JSON);
      if (!pluginJsonPath.toFile().exists()) {
        throw new MojoFailureException(format("At least one of %s or %s is expected", pluginPropertiesPath.toString(),
                                              pluginJsonPath.toString()));
      }
      archiver.addFile(pluginJsonPath.toFile(), MULE_PLUGIN_JSON_LOCATION);
    }
  }

  @Override
  protected void doAddArtifactProperties(File pluginPropertiesFile) throws MojoFailureException {
    if (!pluginPropertiesFile.exists()) {
      return;
    }
    try {
      InputStream is = new FileInputStream(getPluginPropertiesFile());
      Properties loadProps = new Properties();
      loadProps.load(is);
      OutputStream os = new FileOutputStream(pluginPropertiesFile);
      PluginProperties pluginProperties = new PluginProperties(loadProps.getProperty(ARTIFACT_EXPORT_CLASS_PACKAGES),
                                                               loadProps.getProperty(ARTIFACT_EXPORT_RESOURCES),
                                                               join(getPluginDependencies(), ","),
                                                               loadProps.getProperty(PLUGIN_BUNDLE));
      pluginProperties.writeTo(os);
    } catch (IOException e) {
      throw new MojoFailureException("Error copying properties file " + e.getMessage());
    }
  }

  private File getPluginPropertiesFile() {
    if (customPluginProperties.exists()) {
      return customPluginProperties;
    }
    return new File(baseDir, "src/main/resources/" + PLUGIN_PROPERTIES_FILE_NAME);
  }

  @Override
  protected String getArtifactType() {
    return "jar";
  }

  @Override
  protected String getArtifactClassifier() {
    return MULE_PLUGIN_CLASSIFIER;
  }

  @Override
  protected File getMuleArtifactFile() {
    return new File(this.outputDirectory, this.finalName + "-" + MULE_PLUGIN_CLASSIFIER + ".jar");
  }
}
