package org.mule.extension.maven;

import static java.lang.Boolean.parseBoolean;
import static java.lang.String.format;
import static org.mule.extension.maven.loader.MulePluginJsonDescriberLoader.loadMulePluginDescriber;
import org.mule.extension.maven.loader.MavenProjectExtensionModelLoader;
import org.mule.plugin.maven.AbstractMuleMojo;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.extension.api.persistence.ExtensionModelJsonSerializer;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintWriter;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProjectHelper;

/**
 * This Maven Plugin takes the generated resource {@link ExtensionPackageMojo#MULE_PLUGIN_JSON}, to look for the ID on it so that
 * through SPI can generate the {@link ExtensionModel} properly. To do so, the invoker of the plugin must have the required
 * dependencies in its classpath.
 * <p/>
 * Once it generates the {@link ExtensionModel} in memory, it serializes it in and then adds to the current project. If we take
 * the HTTP plugin as sample, the generated JAR artifact will be:
 * 
 * <pre>
 * /Users/someUser/.m2/repository/org/mule/modules/mule-module-http-ext/4.0-SNAPSHOT/mule-module-http-ext-4.0-SNAPSHOT-mule-plugin.jar
 * </pre>
 * 
 * and the generated serialized JSON {@link ExtensionModel} will be:
 * 
 * <pre>
 * /Users/lautar0/.m2/repository/org/mule/modules/mule-module-http-ext/4.0-SNAPSHOT/mule-module-http-ext-4.0-SNAPSHOT-extension-model-4.0.0.json
 * </pre>
 * <p/>
 * This {@link Mojo} must be executed on Mule Plugins that generate {@link ExtensionModel}.
 *
 * @since 1.0
 */
@Mojo(name = "extension-model", defaultPhase = LifecyclePhase.PACKAGE, requiresDependencyResolution = ResolutionScope.TEST)
public class ExtensionModelMojo extends AbstractMuleMojo {

  private static final String TEMPORAL_EXTENSION_MODEL_JSON = "temporal-extension-model.json";

  @Component
  private MavenProjectHelper projectHelper;

  @Override
  public void execute() throws MojoExecutionException, MojoFailureException {
    if (!parseBoolean(System.getProperty("mule.maven.extension.model.disable", "false"))) {
      MulePluginModel mulePluginDescriber = loadMulePluginDescriber(outputDirectory);
      MavenProjectExtensionModelLoader extensionModelLoader = new MavenProjectExtensionModelLoader(getLog());
      final ExtensionModel extensionModel = extensionModelLoader.loadExtension(project, mulePluginDescriber);
      final String serializedExtensionModel = new ExtensionModelJsonSerializer(true).serialize(extensionModel);
      final File generatedExtensionModelFile = new File(outputDirectory, TEMPORAL_EXTENSION_MODEL_JSON);
      try {
        try (PrintWriter out = new PrintWriter(generatedExtensionModelFile)) {
          out.println(serializedExtensionModel);
        }
      } catch (FileNotFoundException e) {
        throw new MojoFailureException(
                                       format("Failure while saving the serialized ExtensionModel to the file [%s]",
                                              generatedExtensionModelFile.getAbsolutePath()));
      }
      projectHelper.attachArtifact(project, "json", "extension-model-" + mulePluginDescriber.getMinMuleVersion(),
                                   generatedExtensionModelFile);
    }
  }
}
