/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.extension.maven.documentation;

import org.mule.extension.maven.documentation.types.TypeFlattener;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.extension.api.dsl.syntax.resolver.DslSyntaxResolver;
import org.mule.runtime.extension.api.dsl.syntax.resolver.SingleExtensionImportTypesStrategy;

import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.Multimap;
import org.apache.commons.io.IOUtils;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.runtime.RuntimeConstants;
import org.apache.velocity.runtime.resource.loader.ClasspathResourceLoader;
import org.asciidoctor.Asciidoctor;
import org.asciidoctor.AttributesBuilder;
import org.asciidoctor.Options;
import org.asciidoctor.OptionsBuilder;
import org.asciidoctor.SafeMode;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;

public class DocumentationGenerator {

  private final static String TEMPLATE = "documentation/main-template.vm";
  private final static String CSS_FILE_NAME = "styles";
  private final static String CSS_EXTENSION = ".css";

  public String generateAsciidoc(ExtensionModel extension) {
    StringWriter writer = new StringWriter();
    VelocityEngine engine = getVelocityEngine();
    VelocityContext context = getVelocityContext(extension);
    engine.getTemplate(TEMPLATE).merge(context, writer);
    return writer.toString();
  }

  public String generateHtml(String asciidoc) {
    Asciidoctor asciidoctor = Asciidoctor.Factory.create();
    File css = getCssFile();
    Options options = OptionsBuilder.options()
        .headerFooter(true).safe(SafeMode.UNSAFE)
        .attributes(AttributesBuilder.attributes()
            .stylesDir(css.getParent())
            .styleSheetName(css.getName())
            .get())
        .get();
    return asciidoctor.render(asciidoc, options);
  }

  private File getCssFile() {
    try {
      File css = File.createTempFile(CSS_FILE_NAME, CSS_EXTENSION);
      copyStreamToFile(getClass().getResourceAsStream("/documentation/" + CSS_FILE_NAME + CSS_EXTENSION), css);
      return css;
    } catch (IOException e) {
      throw new RuntimeException("Error while obtaining css file: " + e.getMessage(), e);
    }
  }

  private VelocityContext getVelocityContext(ExtensionModel extension) {
    VelocityContext context = new VelocityContext();
    DslSyntaxResolver dsl = DslSyntaxResolver.getDefault(extension, new SingleExtensionImportTypesStrategy());
    context.put("label", getLabel(extension));
    context.put("dsl", dsl);
    context.put("sources", getSourcesMap(extension));
    context.put("operations", getOperationsMap(extension));
    context.put("flattener", new TypeFlattener(extension));
    context.put("nameUtil", NameUtils.class);
    context.put("extension", extension);
    return context;
  }

  private String getLabel(ExtensionModel extension) {
    return extension.getConfigurationModels().isEmpty() ? "Module" : "Connector";
  }

  private Multimap<OperationModel, String> getOperationsMap(ExtensionModel extension) {
    Multimap<OperationModel, String> operations = LinkedListMultimap.create();
    extension.getConfigurationModels()
        .forEach(config -> config.getOperationModels().forEach(ope -> operations.put(ope, config.getName())));
    return operations;
  }

  private Multimap<SourceModel, String> getSourcesMap(ExtensionModel extension) {
    Multimap<SourceModel, String> sources = LinkedListMultimap.create();
    extension.getConfigurationModels()
        .forEach(config -> config.getSourceModels().forEach(source -> sources.put(source, config.getName())));
    return sources;
  }

  private VelocityEngine getVelocityEngine() {
    VelocityEngine ve = new VelocityEngine();
    ve.setProperty(RuntimeConstants.RESOURCE_LOADER, "classpath");
    ve.setProperty("runtime.log.logsystem.class", org.apache.velocity.runtime.log.NullLogChute.class.getName());
    ve.setProperty("classpath.resource.loader.class", ClasspathResourceLoader.class.getName());
    ve.init();
    return ve;
  }

  private void copyStreamToFile(InputStream input, File destination) throws IOException {
    if (destination.exists() && !destination.canWrite()) {
      throw new IOException("Destination file does not exist or cannot be written");
    } else {
      try {
        FileOutputStream output = new FileOutputStream(destination);
        try {
          IOUtils.copy(input, output);
        } finally {
          IOUtils.closeQuietly(output);
        }
      } finally {
        IOUtils.closeQuietly(input);
      }
    }
  }
}
