/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.extension.maven;

import static java.io.File.separatorChar;
import static java.lang.String.format;
import static java.util.Collections.sort;
import static org.apache.commons.io.FileUtils.copyFile;
import org.mule.plugin.maven.AbstractMuleMojo;
import org.mule.plugin.maven.ModuleArchiver;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.building.ModelProblem;
import org.apache.maven.model.building.ModelProblem.Severity;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.DefaultProjectBuildingRequest;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.ProjectBuilder;
import org.apache.maven.project.ProjectBuildingException;
import org.apache.maven.project.ProjectBuildingRequest;
import org.apache.maven.project.ProjectBuildingResult;
import org.apache.maven.repository.RepositorySystem;

/**
 * Using the {@link #project} we will clone the current Maven repository needed to execute the plugin into the
 * {@link #outputDirectory}/{@link #REPOSITORY_FOLDER} folder.
 * <p/>
 * This {@link org.apache.maven.plugin.Mojo} not only takes care of the JAR files, but also copies the POMs, respecting the
 * repository structure.
 * <p/>
 * Notice that this class will not copy the current POM file into the repository structure, as it will be part of the
 * {@link ExtensionPackageMojo#addPOMFile(ModuleArchiver)} method
 *
 * @since 1.0
 */
@Mojo(name = "repository-mirror", defaultPhase = LifecyclePhase.VERIFY, requiresDependencyResolution = ResolutionScope.RUNTIME)
public class RepositoryMirrorMojo extends AbstractMuleMojo {

  public static final String REPOSITORY_FOLDER = "repository";

  @Component
  private RepositorySystem repositorySystem;

  @Component
  private ProjectBuilder projectBuilder;

  @Parameter(readonly = true, required = true, defaultValue = "${session}")
  private MavenSession session;

  @Parameter(readonly = true, required = true, defaultValue = "${project.remoteArtifactRepositories}")
  private List<ArtifactRepository> remoteArtifactRepositories;

  @Parameter(readonly = true, required = true, defaultValue = "${localRepository}")
  private ArtifactRepository localRepository;

  private ProjectBuildingRequest projectBuildingRequest;

  @Override
  public void execute() throws MojoExecutionException, MojoFailureException {
    getLog().info(format("Mirroring repository for [%s]", this.project.toString()));
    try {
      initializeProjectBuildingRequest();
      final File repositoryFile = new File(outputDirectory, REPOSITORY_FOLDER);
      if (!repositoryFile.exists()) {
        repositoryFile.mkdirs();
      }
      final Set<Artifact> artifacts = new HashSet<>(project.getArtifacts());
      // Go over the dependencies adding their parent POMs (as well as the dependency's POM)
      for (Artifact dep : new ArrayList<>(artifacts)) {
        addThirdPartyParentPomArtifacts(artifacts, dep);
      }
      // Add current project parent POMs
      addParentPomArtifacts(artifacts);

      // Once gathered all needed artifacts, it will install them in the temporary repository folder
      installArtifacts(repositoryFile, artifacts);
    } catch (Exception e) {
      if (getLog().isDebugEnabled()) {
        getLog().debug(format("There was an exception while building [%s]", project.toString()), e);
      }
      throw e;
    }
  }

  /**
   * Looks for all the parent POMs for the dependency {@code #dep} assuming all POM artifacts to look for are already installed
   * (they must be installed, as they are third party dependencies)
   *
   * @see #addParentPomArtifacts(Set) similar method to resolve POM artifacts for the current project.
   *
   * @param artifacts to store all artifacts found
   * @param dep POM artifact to start looking it up while adding it to the set of artifacts
   * @throws MojoExecutionException if {@link #validatePomArtifactFile(Artifact)} fails validating the {@link File}
   */
  private void addThirdPartyParentPomArtifacts(Set<Artifact> artifacts, Artifact dep) throws MojoExecutionException {
    final MavenProject project = buildProjectFromArtifact(dep);
    addParentDependencyPomArtifacts(project, artifacts);
    final Artifact pomArtifact =
        repositorySystem.createProjectArtifact(dep.getGroupId(), dep.getArtifactId(), dep.getVersion());
    artifacts.add(getResolvedArtifactUsingLocalRepository(pomArtifact));
  }

  /**
   * Goes over every item in the {@code artifacts} collection and copies it to the {@code repositoryFile} destination. If
   * {@code artifacts} is empty, then it creates a marker file to ensure the folder {@link #REPOSITORY_FOLDER} exists in the final
   * JAR file. If this folder doesn't exists, the deployment model on the runtime side must check its existence.
   *
   * @param repositoryFile destination {@link File} to copy the artifacts.
   * @param artifacts {@link Set} of {@link Artifact} to install.
   * @throws MojoExecutionException if there's a problem while copying the artifacts or creating the marker file.
   */
  private void installArtifacts(File repositoryFile, Set<Artifact> artifacts) throws MojoExecutionException {
    final List<Artifact> sortedArtifacts = new ArrayList<>(artifacts);
    sort(sortedArtifacts);
    if (sortedArtifacts.isEmpty()) {
      final File markerFile = new File(repositoryFile, ".marker");
      getLog().info(format("No artifacts to add, adding marker file <%s/%s>", REPOSITORY_FOLDER, markerFile.getName()));
      try {
        markerFile.createNewFile();
      } catch (IOException e) {
        throw new MojoExecutionException(format("The current repository has no artifacts to install, and trying to create [%s] failed",
                                                markerFile.toString()),
                                         e);
      }
    }
    for (Artifact artifact : sortedArtifacts) {
      installArtifact(repositoryFile, artifact);
    }
  }

  /**
   * Resolves both local and remote {@link ArtifactRepository}s, allowing the further fetching of the {@link Artifact}s later on
   * in {@link #buildProjectFromArtifact(Artifact)} through the {@link #projectBuilder}
   */
  private void initializeProjectBuildingRequest() {
    projectBuildingRequest =
        new DefaultProjectBuildingRequest(session.getProjectBuildingRequest());
    projectBuildingRequest.setLocalRepository(localRepository);
    projectBuildingRequest.setRemoteRepositories(remoteArtifactRepositories);
    if (getLog().isDebugEnabled()) {
      getLog().debug(format("Local repository [%s]", projectBuildingRequest.getLocalRepository().getBasedir()));
      projectBuildingRequest.getRemoteRepositories().stream().forEach(artifactRepository -> getLog()
          .debug(format("Remote repository ID [%s], URL [%s]", artifactRepository.getId(), artifactRepository.getUrl())));
    }
  }

  /**
   * Given an {@code artifact} that contains a file, it will copy it's content into the {@code repositoryFile}'s folder respecting
   * the Maven's repository format.
   *
   * @param repositoryFile destination folder to install the artifact
   * @param artifact to copy from
   * @throws MojoExecutionException if the file cannot be copied successfully
   */
  private void installArtifact(File repositoryFile, Artifact artifact) throws MojoExecutionException {
    final File artifactFolderDestination = getFormattedOutputDirectory(repositoryFile, artifact);
    final String artifactFilename = getFormattedFileName(artifact);

    if (!artifactFolderDestination.exists()) {
      artifactFolderDestination.mkdirs();
    }
    final File destinationArtifactFile = new File(artifactFolderDestination, artifactFilename);
    try {
      getLog().info(format("Adding artifact <%s%s>",
                           REPOSITORY_FOLDER,
                           destinationArtifactFile.getAbsolutePath().replaceFirst(Pattern.quote(repositoryFile.getAbsolutePath()),
                                                                                  "")));
      copyFile(artifact.getFile(), destinationArtifactFile);
    } catch (IOException e) {
      throw new MojoExecutionException(format("There was a problem while copying the artifact [%s] file [%s] to the destination [%s]",
                                              artifact.toString(), artifact.getFile().getAbsolutePath(),
                                              destinationArtifactFile.getAbsolutePath()),
                                       e);
    }
  }

  private MavenProject buildProjectFromArtifact(Artifact artifact)
      throws MojoExecutionException {
    MavenProject mavenProject;
    Artifact projectArtifact =
        repositorySystem.createProjectArtifact(artifact.getGroupId(), artifact.getArtifactId(), artifact.getVersion());
    try {
      mavenProject = projectBuilder.build(projectArtifact, projectBuildingRequest).getProject();
    } catch (ProjectBuildingException e) {
      getLog()
          .warn(format("The artifact [%s] seems to have some warnings, enable logs for more information", artifact.toString()));
      if (getLog().isDebugEnabled()) {
        getLog().warn(format("The artifact [%s] had the following issue ", artifact.toString()), e);
      }
      // If the current pom has failures we will dismiss them as there's at least a successful result
      if (e.getResults() == null || e.getResults().size() != 1) {
        throw new MojoExecutionException(format("There was an issue while trying to create a maven project from the artifact [%s]",
                                                artifact.toString()),
                                         e);
      }
      final ProjectBuildingResult projectBuildingResult = e.getResults().get(0);
      final List<ModelProblem> collect = projectBuildingResult.getProblems().stream()
          .filter(modelProblem -> modelProblem.getSeverity().equals(Severity.FATAL)).collect(
                                                                                             Collectors.toList());
      if (!collect.isEmpty()) {
        throw new MojoExecutionException(format("There was an issue while trying to create a maven project from the artifact [%s], several FATAL errors were found",
                                                artifact.toString()),
                                         e);
      }
      // If we reach here it means all the problems were acceptable
      mavenProject = projectBuildingResult.getProject();
    }
    return mavenProject;
  }

  /**
   * Looks for all the parent POMs for the {@code projectDependency} *relying* in the current status of the repository, as the
   * {@link #project}'s dependencies must have been installed. Because of that, this method relies hevily on
   * {@link #getResolvedArtifactUsingLocalRepository(Artifact)}
   *
   * @param artifacts to store all artifacts found
   * @throws MojoExecutionException if {@link #validatePomArtifactFile(Artifact)} fails validating the {@link File}
   */
  private void addParentDependencyPomArtifacts(MavenProject projectDependency, Set<Artifact> artifacts)
      throws MojoExecutionException {
    // We skip the first POM parent, as it will be used when copying the current artifact's project parent's POMs files only
    MavenProject currentProject = projectDependency;
    while (currentProject.hasParent()) {
      currentProject = currentProject.getParent();
      final Artifact pomArtifact = currentProject.getArtifact();
      if (!artifacts.add(getResolvedArtifactUsingLocalRepository(pomArtifact))) {
        // Artifact already in the set, no more parents needed to add
        break;
      }
    }
  }

  /**
   * Looks for all the parent POMs for the current {@link #project} without relying in the current status of the repository, as
   * the parent might not have been installed yet, thus it uses the current {@link File} that's targeting the
   * {@link MavenProject#getFile()} method.
   * <p/>
   * If it happens that {@link MavenProject#getFile()} returns {@code null}, it implies that the POM that's being targeted is
   * actually a third party dependency that must be looked into the repository, falling back to the
   * {@link #getResolvedArtifactUsingLocalRepository(Artifact)} method.
   *
   * @param artifacts to store all artifacts found
   * @throws MojoExecutionException if {@link #validatePomArtifactFile(Artifact)} fails validating the {@link File}
   */
  private void addParentPomArtifacts(Set<Artifact> artifacts)
      throws MojoExecutionException {
    // We skip the first POM parent, as it will be used when copying the current artifact's project parent's POMs files only
    MavenProject currentProject = project;
    boolean projectParent = true;
    while (currentProject.hasParent() && projectParent) {
      currentProject = currentProject.getParent();
      if (currentProject.getFile() == null) {
        projectParent = false;
      } else {
        Artifact pomArtifact = currentProject.getArtifact();
        pomArtifact.setFile(currentProject.getFile());
        validatePomArtifactFile(pomArtifact);
        if (!artifacts.add(pomArtifact)) {
          // Artifact already in the set, no more parents needed to add
          break;
        }
      }
    }
    // if projectParent=false, there are POMs needed to be resolved using the dependency parent mechanism as they don't belong to
    // the current project
    if (!projectParent) {
      final Artifact unresolvedParentPomArtifact = currentProject.getArtifact();
      addThirdPartyParentPomArtifacts(artifacts, unresolvedParentPomArtifact);
    }
  }

  private Artifact getResolvedArtifactUsingLocalRepository(Artifact pomArtifact) throws MojoExecutionException {
    final Artifact resolvedPomArtifact = localRepository.find(pomArtifact);
    validatePomArtifactFile(resolvedPomArtifact);
    return resolvedPomArtifact;
  }

  /**
   * Validates the parametrized {@code resolvedPomArtifact} has a valid an existing file under {@link Artifact#getFile()}.
   *
   * @param resolvedPomArtifact an artifact that should have been already resolved and it only misses the file validation
   * @throws MojoExecutionException if the file of the parametrized {@code resolvedPomArtifact} is null or id does not exists.
   */
  private void validatePomArtifactFile(Artifact resolvedPomArtifact) throws MojoExecutionException {
    if (resolvedPomArtifact.getFile() == null) {
      throw new MojoExecutionException(format("There was a problem trying to resolve the artifact's file location for [%s], file was null",
                                              resolvedPomArtifact.toString()));
    }
    if (!resolvedPomArtifact.getFile().exists()) {
      throw new MojoExecutionException(format("There was a problem trying to resolve the artifact's file location for [%s], file [%s] doesn't exists",
                                              resolvedPomArtifact.toString(), resolvedPomArtifact.getFile().getAbsolutePath()));
    }
  }

  private String getFormattedFileName(Artifact artifact) {
    StringBuilder destFileName = new StringBuilder();
    String versionString = "-" + getNormalizedVersion(artifact);
    String classifierString = "";

    if (artifact.getClassifier() != null && !artifact.getClassifier().isEmpty()) {
      classifierString = "-" + artifact.getClassifier();
    }
    destFileName.append(artifact.getArtifactId()).append(versionString);
    destFileName.append(classifierString).append(".");
    destFileName.append(artifact.getArtifactHandler().getExtension());

    return destFileName.toString();
  }

  /**
   * This method is worthwhile if and only if the current artifact is a SNAPSHOT version but the file that's pointing has the
   * timestamp.
   * <p/>
   * This means that if the current {@code artifact} has a version ({@link Artifact#getVersion()}) similar to
   * "20170103.185645-1273" this method will return the base version of it ({@link Artifact#getBaseVersion()}, which usually maps
   * to "SNAPSHOT"), allowing it's further consumption in the deployment model without having to ship the maven-metadata.xml files
   * in the internal mirrored repository.
   *
   * @param artifact to validate if it's SNAPSHOT or not.
   * @return the normalized artifact's version.
   */
  private String getNormalizedVersion(Artifact artifact) {
    if (artifact.isSnapshot() && !artifact.getVersion().equals(artifact.getBaseVersion())) {
      return artifact.getBaseVersion();
    }
    return artifact.getVersion();
  }

  private static File getFormattedOutputDirectory(File outputDirectory, Artifact artifact) {
    StringBuilder sb = new StringBuilder(128);
    // group id
    sb.append(artifact.getGroupId().replace('.', separatorChar)).append(separatorChar);
    // artifact id
    sb.append(artifact.getArtifactId()).append(separatorChar);
    // version
    sb.append(artifact.getBaseVersion()).append(separatorChar);

    return new File(outputDirectory, sb.toString());
  }
}
