/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extension.maven;

import static java.lang.Boolean.parseBoolean;
import static java.lang.String.format;
import static org.apache.maven.plugins.annotations.LifecyclePhase.VERIFY;
import static org.apache.maven.plugins.annotations.ResolutionScope.TEST;
import static org.mule.extension.maven.loader.MulePluginJsonDescriberLoader.loadMulePluginDescriber;
import org.mule.extension.maven.documentation.DocumentationGenerator;
import org.mule.extension.maven.loader.MavenProjectExtensionModelLoader;
import org.mule.plugin.maven.AbstractMuleMojo;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.meta.model.ExtensionModel;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintWriter;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectHelper;
import org.zeroturnaround.zip.ZipUtil;

/**
 * A simple maven plugin that generates Asciidoc and HTML documentation pages for an extension.
 */
@Mojo(name = "documentation", defaultPhase = VERIFY, requiresDependencyResolution = TEST)
public class ExtensionDocumentationMojo extends AbstractMuleMojo {

  private static final String DOC_SUFFIX = "-documentation";
  private static final String ZIP = "zip";

  /**
   * The maven project where this MOJO is being executed.
   */
  @Parameter(defaultValue = "${project}", required = true)
  private MavenProject project;

  /**
   * The output directory where the temporal generated documentation files are going to be located.
   */
  @Parameter(defaultValue = "${project.build.directory}/temporal-docs", required = true)
  private File outputDir;

  @Component
  private MavenProjectHelper projectHelper;

  private static final DocumentationGenerator documentationGenerator = new DocumentationGenerator();

  public void execute() throws MojoExecutionException, MojoFailureException {
    if (!parseBoolean(System.getProperty("mule.maven.extension.model.disable", "false"))) {
      if (!outputDir.exists()) {
        outputDir.mkdirs();
      }
      MulePluginModel mulePluginModel = loadMulePluginDescriber(outputDirectory);
      MavenProjectExtensionModelLoader extensionModelLoader = new MavenProjectExtensionModelLoader(getLog());
      ExtensionModel extension = extensionModelLoader.loadExtension(project, mulePluginModel);
      String asciidoc = documentationGenerator.generateAsciidoc(extension);
      String html = documentationGenerator.generateHtml(asciidoc);
      generateFile(getSanitizedName(extension) + DOC_SUFFIX + ".adoc", asciidoc);
      generateFile(getSanitizedName(extension) + DOC_SUFFIX + ".html", html);
      File zip = createDocsZipArtifact();
      projectHelper.attachArtifact(project, ZIP, "extension" + DOC_SUFFIX, zip);
    }
  }

  private File createDocsZipArtifact() {
    File packagedZipFile = new File(outputDir.toString() + "." + ZIP);
    ZipUtil.pack(outputDir, packagedZipFile);
    return packagedZipFile;
  }

  private String getSanitizedName(ExtensionModel extensionModel) {
    return extensionModel.getName().toLowerCase().replace(" ", "-");
  }

  private void generateFile(String fileName, String content) throws MojoExecutionException {
    File file = new File(outputDir, fileName);
    try {
      PrintWriter printWriter = new PrintWriter(file);
      printWriter.print(content);
      printWriter.close();
    } catch (FileNotFoundException e) {
      throw new MojoExecutionException(
                                       format("Cannot write the file [%s] (it does not exist), cannot generate documentation",
                                              file.getAbsolutePath()),
                                       e);
    }
    getLog().info(format("Generated file [%s]", file.getAbsolutePath()));
  }
}
