/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extension.maven.documentation.types;

import static java.util.Collections.unmodifiableMap;
import static java.util.stream.Collectors.toList;
import static org.mule.extension.maven.documentation.types.ObjectTypeUtils.isMessage;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.UnionType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.operation.HasOperationModels;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.source.HasSourceModels;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.meta.model.util.ExtensionWalker;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

class TypeRegister {

  private final Map<String, ObjectType> types = new LinkedHashMap<>();
  private final Map<String, List<String>> subtypesIdMap = new LinkedHashMap<>();

  TypeRegister(ExtensionModel extension) {
    fillSubtypes(extension);
    fillExposedTypes(extension);
  }

  private void fillSubtypes(ExtensionModel extension) {
    extension.getSubTypes()
        .forEach(subtype -> {
          List<String> subtypesIds = subtype.getSubTypes().stream()
              .filter(o -> ObjectTypeUtils.getObjectTypeId(o).isPresent())
              .map(o -> ObjectTypeUtils.getObjectTypeId(o).get()).collect(toList());
          ObjectTypeUtils.getObjectTypeId(subtype.getBaseType())
              .ifPresent(baseTypeId -> subtypesIdMap.put(baseTypeId, subtypesIds));
        });
  }

  private void fillExposedTypes(ExtensionModel extension) {
    MetadataTypeVisitor visitor = new RegisterVisitor();

    new ExtensionWalker() {

      @Override
      protected void onSource(HasSourceModels owner, SourceModel source) {
        visitComponent(source);
      }

      @Override
      protected void onOperation(HasOperationModels owner, OperationModel operation) {
        visitComponent(operation);
      }

      private void visitComponent(ComponentModel component) {
        component.getOutput().getType().accept(visitor);
        component.getOutputAttributes().getType().accept(visitor);
      }

      @Override
      protected void onParameter(ParameterizedModel owner, ParameterGroupModel groupModel, ParameterModel parameter) {
        parameter.getType().accept(visitor);
      }
    }.walk(extension);

    extension.getSubTypes().forEach(subtype -> subtype.getSubTypes().forEach(t -> t.accept(visitor)));
  }

  Map<String, ObjectType> getTypes() {
    return unmodifiableMap(types);
  }

  Map<String, List<String>> getSubtypes() {
    return unmodifiableMap(subtypesIdMap);
  }

  private void addType(ObjectType type) {
    ObjectTypeUtils.getObjectTypeId(type).ifPresent(id -> {
      if (!subtypesIdMap.containsKey(id) && !type.getFields().isEmpty()) {
        types.put(id, type);
      }
    });
  }

  private class RegisterVisitor extends MetadataTypeVisitor {

    @Override
    public void visitArrayType(ArrayType arrayType) {
      arrayType.getType().accept(this);
    }

    @Override
    public void visitObject(ObjectType object) {
      if (isMessage(object)) {
        object.getFields().forEach(f -> f.getValue().accept(this));
      } else {
        ObjectTypeUtils.getObjectTypeId(object).ifPresent(id -> {
          if (!types.containsKey(id)) {
            addType(object);
            object.getFields().forEach(f -> f.getValue().accept(this));
          }
        });
      }
    }

    @Override
    public void visitUnion(UnionType unionType) {
      unionType.getTypes().forEach(t -> t.accept(this));
    }
  }
}
