/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extension.maven.documentation.types;

import static java.lang.String.format;
import static org.mule.extension.maven.documentation.NameUtils.prettify;
import static org.mule.metadata.api.utils.MetadataTypeUtils.getLocalPart;
import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;
import static org.mule.runtime.extension.api.declaration.type.TypeUtils.isInfrastructure;
import org.mule.metadata.api.annotation.DefaultValueAnnotation;
import org.mule.metadata.api.annotation.TypeAliasAnnotation;
import org.mule.metadata.api.model.ObjectType;

import java.util.Optional;
import java.util.StringJoiner;

class ObjectTypeUtils {

  public static final String EMPTY = "";

  static Optional<String> getObjectTypeId(ObjectType objectType) {
    Optional<String> id = getTypeId(objectType);
    Optional<String> alias = objectType.getAnnotation(TypeAliasAnnotation.class).map(TypeAliasAnnotation::getValue);
    if (id.filter(value -> value.startsWith("javax.") || value.startsWith("java.")).isPresent()) {
      return isInfrastructure(objectType) && alias.isPresent() ? alias : Optional.empty();
    }
    return alias.isPresent() ? alias : id.map(v -> v.contains(".") ? v.substring(v.lastIndexOf(".") + 1) : v);
  }

  static String toAsciidoc(ObjectType type, TypeFlattener flattener) {
    StringJoiner stringJoiner = new StringJoiner("\n");
    stringJoiner.add("[cols=\".^30%,.^40%,.^30%\", options=\"header\"]");
    stringJoiner.add("|======================");
    stringJoiner.add("| Field | Type | Default Value");
    type.getFields().forEach(field -> {

      String localPart = getLocalPart(field);
      String defaultValue = field.getAnnotation(DefaultValueAnnotation.class)
          .map(DefaultValueAnnotation::getValue)
          .orElse(EMPTY);

      stringJoiner.add(format("| %s a| %s | %s", prettify(localPart), flattener.flat(field.getValue()), defaultValue));
    });
    stringJoiner.add("|======================");
    return stringJoiner.toString();
  }

  static boolean isMessage(ObjectType objectType) {
    Optional<String> typeId = getTypeId(objectType);
    return typeId.isPresent() && typeId.get().equals(org.mule.runtime.api.message.Message.class.getName());
  }
}
