/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extension.maven.documentation.types;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import org.mule.metadata.api.annotation.DefaultValueAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.java.api.JavaTypeLoader;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.extension.api.model.ImmutableOutputModel;

import org.junit.Before;
import org.junit.Test;

import java.util.Map;


public class TypeFlattenerTestCase {

  private final static ObjectType TYPE_WITH_DEFAULT_VALUE = getTypeWithDefaultValue();
  private final static ObjectType TYPE_WITH_DESCRIPTION = getTypeWithDescriptions();
  private final static ObjectType TYPE_WITH_REQUIRED_FIELDS = getTypeWithRequiredFields();

  private JavaTypeLoader loader = new JavaTypeLoader(Thread.currentThread().getContextClassLoader());
  private TypeFlattener flattener;

  @Before
  public void setup() {
    ExtensionModel extensionModel = mock(ExtensionModel.class);
    OperationModel operation = mock(OperationModel.class);
    when(operation.getOutput()).thenReturn(new ImmutableOutputModel("", loader.load(ComplexType.class), true, null));
    when(operation.getOutputAttributes()).thenReturn(new ImmutableOutputModel("", loader.load(RecursiveType.class), true, null));
    ParameterModel complex = getMockedParameter(loader.load(ComplexType.class));
    ParameterModel recursive = getMockedParameter(loader.load(RecursiveType.class));
    ParameterModel enumParam = getMockedParameter(loader.load(EnumType.class));
    ParameterModel typeWithDefaultValueParam = getMockedParameter(TYPE_WITH_DEFAULT_VALUE);
    ParameterModel typeWithDescriptions = getMockedParameter(TYPE_WITH_DESCRIPTION);
    ParameterModel typeWithRequiredFields = getMockedParameter(TYPE_WITH_REQUIRED_FIELDS);
    ParameterGroupModel group = mock(ParameterGroupModel.class);
    when(group.getParameterModels()).thenReturn(asList(complex,
                                                       recursive,
                                                       enumParam,
                                                       typeWithDefaultValueParam,
                                                       typeWithDescriptions,
                                                       typeWithRequiredFields));
    when(operation.getParameterGroupModels()).thenReturn(singletonList(group));
    when(extensionModel.getOperationModels()).thenReturn(singletonList(operation));
    this.flattener = new TypeFlattener(extensionModel);
  }

  @Test
  public void recursiveType() {
    MetadataType type = loader.load(RecursiveType.class);
    String flat = flattener.flat(type);
    assertThat(flat, is("<<RecursiveType>>"));
    Map<String, String> registeredTypes = flattener.getObjectTypes();
    assertThat(registeredTypes.get("RecursiveType"), is("[cols=\".^20%,.^25%,.^30%,.^15%,.^10%\", options=\"header\"]\n"
        + "|======================\n"
        + "| Field | Type | Description | Default Value | Required\n"
        + "| Delegate a| <<RecursiveType>> |  |  | \n"
        + "| Field a| String |  |  | \n"
        + "|======================"));
  }

  @Test
  public void nestedType() {
    MetadataType type = loader.load(ComplexType.class);
    String flat = flattener.flat(type);
    assertThat(flat, is("<<ComplexType>>"));
    Map<String, String> registeredTypes = flattener.getObjectTypes();
    assertThat(registeredTypes.get("ComplexType"), is("[cols=\".^20%,.^25%,.^30%,.^15%,.^10%\", options=\"header\"]\n"
        + "|======================\n"
        + "| Field | Type | Description | Default Value | Required\n"
        + "| Recursive Type a| <<RecursiveType>> |  |  | \n"
        + "|======================"));
  }

  @Test
  public void withDefaultValue() {
    String flat = flattener.flat(TYPE_WITH_DEFAULT_VALUE);
    assertThat(flat, is("<<TIDDefaultValue>>"));
    Map<String, String> registeredTypes = flattener.getObjectTypes();
    assertThat(registeredTypes.get("TIDDefaultValue"), is("[cols=\".^20%,.^25%,.^30%,.^15%,.^10%\", options=\"header\"]\n"
        + "|======================\n"
        + "| Field | Type | Description | Default Value | Required\n"
        + "| Field A a| String |  | DefaultVal | \n"
        + "|======================"));
  }

  @Test
  public void withDescriptionValue() {
    String flat = flattener.flat(TYPE_WITH_DESCRIPTION);
    assertThat(flat, is("<<TIDDescription>>"));
    Map<String, String> registeredTypes = flattener.getObjectTypes();
    assertThat(registeredTypes.get("TIDDescription"), is("[cols=\".^20%,.^25%,.^30%,.^15%,.^10%\", options=\"header\"]\n"
        + "|======================\n"
        + "| Field | Type | Description | Default Value | Required\n"
        + "| Field A a| String | Descriptive Description |  | \n"
        + "|======================"));
  }

  @Test
  public void withRequiredField() {
    String flat = flattener.flat(TYPE_WITH_REQUIRED_FIELDS);
    assertThat(flat, is("<<TIDRequiredField>>"));
    Map<String, String> registeredTypes = flattener.getObjectTypes();
    assertThat(registeredTypes.get("TIDRequiredField"), is("[cols=\".^20%,.^25%,.^30%,.^15%,.^10%\", options=\"header\"]\n"
        + "|======================\n"
        + "| Field | Type | Description | Default Value | Required\n"
        + "| Field A a| String |  |  | x\n"
        + "| Field B a| String |  |  | \n"
        + "|======================"));
  }


  @Test
  public void enumType() {
    MetadataType type = loader.load(EnumType.class);
    String flat = flattener.flat(type);
    assertThat(flat, is("Enumeration, one of:\n\n** First\n** Second\n** Third"));
  }

  @Test
  public void plainObject() {
    MetadataType type = loader.load(Object.class);
    String flat = flattener.flat(type);
    assertThat(flat, is("Any"));
  }

  @Test
  public void arrayType() {
    MetadataType recursive = loader.load(RecursiveType.class);
    ArrayType array = BaseTypeBuilder.create(JAVA).arrayType().of(recursive).build();
    String flattened = flattener.flat(array);
    assertThat(flattened, is("Array of <<RecursiveType>>"));
  }

  private ParameterModel getMockedParameter(MetadataType type) {
    ParameterModel param = mock(ParameterModel.class);
    when(param.getType()).thenReturn(type);
    return param;
  }

  private static ObjectType getTypeWithRequiredFields() {
    ObjectTypeBuilder builder = BaseTypeBuilder.create(JAVA).objectType().with(new TypeIdAnnotation("TIDRequiredField"));
    ObjectFieldTypeBuilder a = builder.addField();
    a.key("Field A");
    a.value().stringType();
    a.required(true);

    ObjectFieldTypeBuilder b = builder.addField();
    b.key("Field B");
    b.value().stringType();
    b.required(false);
    return builder.build();
  }

  private static ObjectType getTypeWithDescriptions() {
    ObjectTypeBuilder builder = BaseTypeBuilder.create(JAVA)
        .objectType()
        .with(new TypeIdAnnotation("TIDDescription"))
        .description("Description");
    ObjectFieldTypeBuilder field = builder.addField();
    field.key("Field A");
    field.value().stringType();
    field.description("Descriptive Description");
    return builder.build();
  }

  private static ObjectType getTypeWithDefaultValue() {
    ObjectTypeBuilder builder = BaseTypeBuilder.create(JAVA)
        .objectType()
        .with(new TypeIdAnnotation("TIDDefaultValue"))
        .description("Description");
    ObjectFieldTypeBuilder field = builder.addField();
    field.key("Field A");
    field.value().stringType();
    field.with(new DefaultValueAnnotation("DefaultVal"));
    return builder.build();
  }

  enum EnumType {
    First, Second, Third
  }

  class ComplexType {

    private RecursiveType recursiveType;

    public RecursiveType getRecursiveType() {
      return recursiveType;
    }
  }

  class RecursiveType {

    private String field;

    private RecursiveType delegate;

    public RecursiveType getDelegate() {
      return delegate;
    }

    public String getField() {
      return field;
    }
  }
}
