/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.extension.maven.documentation;

import static java.lang.String.format;
import static org.mule.extension.maven.documentation.internal.model.XmlExtensionDocumentation.EXTENSION_DESCRIPTIONS_FILE_RELATIVE_PATH_MASK;
import org.mule.extension.maven.documentation.internal.model.XmlExtensionDocumentation;
import org.mule.extension.maven.documentation.types.TypeFlattener;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.extension.api.dsl.syntax.resolver.DslSyntaxResolver;
import org.mule.runtime.extension.api.dsl.syntax.resolver.SingleExtensionImportTypesStrategy;
import org.mule.runtime.extension.internal.GenericXmlSerializer;

import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.Multimap;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.nio.file.Files;

import org.apache.commons.io.IOUtils;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.runtime.RuntimeConstants;
import org.apache.velocity.runtime.resource.loader.ClasspathResourceLoader;
import org.asciidoctor.Asciidoctor;
import org.asciidoctor.AttributesBuilder;
import org.asciidoctor.Options;
import org.asciidoctor.OptionsBuilder;
import org.asciidoctor.SafeMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DocumentationGenerator {

  private static final Logger LOGGER = LoggerFactory.getLogger(DocumentationGenerator.class);
  private final static String TEMPLATE = "documentation/main-template.vm";
  private final static String CSS_FILE_NAME = "styles";
  private final static String CSS_EXTENSION = ".css";
  private static final String ABSENT_CONFIGURATION_NAME = "_ABSENT_CONFIGURATION_ID_";
  private XmlExtensionDocumentation xmlDocumentation;

  public DocumentationGenerator(ExtensionModel extension, File outputDirectory) {
    try {
      GenericXmlSerializer<XmlExtensionDocumentation> deserializer = new GenericXmlSerializer<>(XmlExtensionDocumentation.class);
      String name = extension.getName().replace(" ", "-").toLowerCase();
      final File xmlDocumentationFile = new File(outputDirectory, format(EXTENSION_DESCRIPTIONS_FILE_RELATIVE_PATH_MASK, name));
      final String serializedxmlDoc = new String(Files.readAllBytes(xmlDocumentationFile.toPath()));
      xmlDocumentation = deserializer.deserialize(serializedxmlDoc);
    } catch (Exception e) {
      LOGGER.debug("An error occurred while loading the Extension descriptions file", e);
    }
  }

  public String generateAsciidoc(ExtensionModel extension) {
    StringWriter writer = new StringWriter();
    VelocityEngine engine = getVelocityEngine();
    VelocityContext context = getVelocityContext(extension);
    engine.getTemplate(TEMPLATE).merge(context, writer);
    return writer.toString();
  }

  public String generateHtml(String asciidoc) {
    Asciidoctor asciidoctor = Asciidoctor.Factory.create();
    File css = getCssFile();
    Options options = OptionsBuilder.options()
        .headerFooter(true).safe(SafeMode.UNSAFE)
        .attributes(AttributesBuilder.attributes()
            .stylesDir(css.getParent())
            .styleSheetName(css.getName())
            .get())
        .get();
    return asciidoctor.render(asciidoc, options);
  }

  private File getCssFile() {
    try {
      File css = File.createTempFile(CSS_FILE_NAME, CSS_EXTENSION);
      copyStreamToFile(getClass().getResourceAsStream("/documentation/" + CSS_FILE_NAME + CSS_EXTENSION), css);
      return css;
    } catch (IOException e) {
      throw new RuntimeException("Error while obtaining css file: " + e.getMessage(), e);
    }
  }

  private VelocityContext getVelocityContext(ExtensionModel extension) {
    VelocityContext context = new VelocityContext();
    DslSyntaxResolver dsl = DslSyntaxResolver.getDefault(extension, new SingleExtensionImportTypesStrategy());
    context.put("label", getLabel(extension));
    context.put("dsl", dsl);
    context.put("sources", getSourcesMap(extension));
    context.put("operations", getOperationsMap(extension));
    context.put("flattener", new TypeFlattener(extension, xmlDocumentation));
    context.put("nameUtil", NameUtils.class);
    context.put("extension", extension);
    context.put("absentConfiguration", ABSENT_CONFIGURATION_NAME);
    return context;
  }

  private String getLabel(ExtensionModel extension) {
    return extension.getConnectionProviders().isEmpty() ? "Module" : "Connector";
  }

  private Multimap<OperationModel, String> getOperationsMap(ExtensionModel extension) {
    Multimap<OperationModel, String> operations = LinkedListMultimap.create();
    extension.getConfigurationModels()
        .forEach(config -> config.getOperationModels().forEach(ope -> operations.put(ope, config.getName())));
    extension.getOperationModels().forEach(o -> operations.put(o, ABSENT_CONFIGURATION_NAME));
    return operations;
  }

  private Multimap<SourceModel, String> getSourcesMap(ExtensionModel extension) {
    Multimap<SourceModel, String> sources = LinkedListMultimap.create();
    extension.getConfigurationModels()
        .forEach(config -> config.getSourceModels().forEach(source -> sources.put(source, config.getName())));
    return sources;
  }

  private VelocityEngine getVelocityEngine() {
    VelocityEngine ve = new VelocityEngine();
    ve.setProperty(RuntimeConstants.RESOURCE_LOADER, "classpath");
    ve.setProperty("runtime.log.logsystem.class", org.apache.velocity.runtime.log.NullLogChute.class.getName());
    ve.setProperty("classpath.resource.loader.class", ClasspathResourceLoader.class.getName());
    ve.init();
    return ve;
  }

  private void copyStreamToFile(InputStream input, File destination) throws IOException {
    if (destination.exists() && !destination.canWrite()) {
      throw new IOException("Destination file does not exist or cannot be written");
    } else {
      try {
        FileOutputStream output = new FileOutputStream(destination);
        try {
          IOUtils.copy(input, output);
        } finally {
          IOUtils.closeQuietly(output);
        }
      } finally {
        IOUtils.closeQuietly(input);
      }
    }
  }
}
