package org.mule.extension.maven;

import static com.google.common.base.Preconditions.checkArgument;
import static java.lang.String.format;
import static java.nio.file.Files.exists;
import static java.nio.file.Files.notExists;
import static java.nio.file.Files.readAllBytes;
import static java.nio.file.Files.write;
import static java.util.Collections.emptyMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static jdk.nashorn.internal.objects.NativeArray.join;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.commons.io.FileUtils.listFiles;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.uncapitalize;
import static org.apache.maven.plugins.annotations.LifecyclePhase.COMPILE;
import static org.mule.extension.maven.loader.MulePluginJsonDescriberLoader.loadMulePluginDescriber;
import static org.mule.runtime.api.deployment.meta.Product.MULE;
import static org.mule.runtime.api.deployment.meta.Product.MULE_EE;
import static org.mule.runtime.api.util.Preconditions.checkState;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import org.apache.commons.io.filefilter.FileFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.mule.plugin.maven.AbstractMuleMojo;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptor;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.deployment.persistence.MulePluginModelJsonSerializer;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.extension.api.annotation.Extension;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * Mojo responsible of looking for an existing {@link ExtensionPackageMojo#MULE_ARTIFACT_JSON} in META-INF folder.
 * <p/>
 * If it doesn't exists, it assumes it's a <module/> scenario, to which it will try to generate one with the following
 * conventions:
 * <ol>
 * <li>Will scan the /classes looking for anything that starts with {@link #PREFIX_SMART_CONNECTOR_NAME} and ends with
 * {@link #SUFFIX_SMART_CONNECTOR_NAME}</li>
 * <li>For each element found, it will open it and look if they are valid XML files and their root elements matches to
 * {@link #MODULE_ROOT_ELEMENT}</li>
 * <li>There must be one, and only one, file in the /classes that match the previous rules. If not, fails.</li>
 * </ol>
 *
 * @since 1.0
 */
@Mojo(name = "extension-descriptor", defaultPhase = COMPILE, threadSafe = true)
public class ExtensionDescriptorMojo extends AbstractMuleMojo {

  private static final String META_INF = "META-INF";
  private static final String AUTO_GENERATED_MULE_ARTIFACT_DESCRIPTOR =
      Paths.get(META_INF, "auto-generated-" + MULE_ARTIFACT_JSON).toString();

  private static final String MULE_LOADER_ID = "mule";
  public static final String XML_BASED_EXTENSION_MODEL_LOADER = "xml-based";
  public static final String RESOURCE_XML = "resource-xml";
  public static final String VALIDATE_XML = "validate-xml";
  private static final String PREFIX_SMART_CONNECTOR_NAME = "module-";
  private static final String SUFFIX_SMART_CONNECTOR_NAME = ".xml";
  private static final String MODULE_ROOT_ELEMENT = "module";
  private static final String NAME_ATTRIBUTE = "name";
  private static final String MIN_MULE_VERSION_ATTRIBUTE = "minMuleVersion";
  private static final String MIN_MULE_VERSION_DESCRIPTOR_PROPERTY = "minMuleVersion";
  private static final String REQUIRED_PRODUCT_DESCRIPTOR_PROPERTY = "requiredProduct";
  private static final String MULE_VERSION_PROPERTY = "mule.version";
  private static final String MULE_CE_PRODUCT = "MULE";
  private static final String MULE_EE_PRODUCT = "MULE_EE";
  public static final String SNAPSHOT_SUFFIX = "SNAPSHOT";

  /**
   * Returns a {@link Path} to an existing descriptor file or fails.
   *
   * @param outputDirectory to look for the current output directory
   * @return an existing {@link Path} to a descriptor file
   * @throws MojoFailureException if the descriptor file is absent (probably because this Mojo hasn't been executed)
   */
  public static Path descriptorPathOrFail(File outputDirectory) throws MojoFailureException {
    return descriptorPath(outputDirectory, "classes" + File.separator + ExtensionPackageMojo.MULE_PLUGIN_JSON_JAR_DESTINATION);
  }

  /**
   * Returns a {@link Path} to an auto generated descriptor.
   *
   * @param outputDirectory to look for the current output directory
   * @return an existing {@link Path} to a descriptor
   */
  public static Path autoGeneratedDescriptorPath(File outputDirectory) throws MojoFailureException {
    return Paths.get(outputDirectory.getAbsolutePath(), AUTO_GENERATED_MULE_ARTIFACT_DESCRIPTOR);
  }

  private static Path descriptorPath(File outputDirectory, String descriptorLocation) throws MojoFailureException {
    final Path path = Paths.get(outputDirectory.getAbsolutePath(), descriptorLocation);
    if (notExists(path)) {
      throw new MojoFailureException(format("Should not have reach this point, could not obtain descriptor file from [%s]",
                                            path));
    }
    return path;
  }

  @Override
  public void execute() throws MojoExecutionException, MojoFailureException {
    // This file may be created by MulePluginDescriptorGenerator
    final Path autoGeneratedDescriptorPath = autoGeneratedDescriptorPath(new File(project.getBuild().getOutputDirectory()));
    final Path targetDescriptorPath =
        Paths.get(project.getBuild().getOutputDirectory(), META_INF, MULE_ARTIFACT, MULE_ARTIFACT_JSON);
    if (!exists(autoGeneratedDescriptorPath) && !exists(targetDescriptorPath)) {
      getLog().debug(format("No [%s] descriptor found, trying to create one", targetDescriptorPath));
      try {
        createDescriptor(autoGeneratedDescriptorPath);
      } catch (MojoExecutionException e) {
        if (!exists(targetDescriptorPath)) {
          // continue if there's a manual defined descriptor.
          throw e;
        }
      }
    }
    applyUserCustomizedPropertiesAndDefaultValues(targetDescriptorPath, autoGeneratedDescriptorPath);
    MulePluginModel mulePluginModel = loadMulePluginDescriber(outputDirectory);
    validate(mulePluginModel);
    if (exists(autoGeneratedDescriptorPath)) {
      deleteQuietly(autoGeneratedDescriptorPath.toFile());
    }
  }

  private void validate(MulePluginModel mulePluginModel) {
    mulePluginModel.getLicense().ifPresent(licenseModel -> {
      licenseModel.getRequiredEntitlement().ifPresent(requiredEntitlement -> {
        String provider = licenseModel.getProvider();
        if (!Extension.MULESOFT.equals(provider)) {
          String outputDirectory = project.getBuild().getOutputDirectory();
          String keyFileName = String.format("%s-%s.key", provider, uncapitalize(mulePluginModel.getName().replace(" ", "")));
          if (!new File(outputDirectory, keyFileName).exists()) {
            throw new RuntimeException(format(
                                              "The plugin requires an entitlement but there is no key file (%s) found. The provider key file for customer licenses must be added to the classpath at the root level",
                                              keyFileName));
          }
        }
      });
    });
  }

  private void applyUserCustomizedPropertiesAndDefaultValues(Path targetDescriptorPath, Path autoGeneratedDescriptorPath)
      throws MojoExecutionException {
    String minMuleVersion = (String) project.getProperties().get(MULE_VERSION_PROPERTY);
    String requiredProduct = MULE_CE_PRODUCT;

    JsonElement targetDescriptorJsonElement;
    if (exists(autoGeneratedDescriptorPath)) {
      targetDescriptorJsonElement = getJsonElement(autoGeneratedDescriptorPath);
    } else {
      targetDescriptorJsonElement = getJsonElement(targetDescriptorPath);
    }

    if (targetDescriptorJsonElement.getAsJsonObject().has(REQUIRED_PRODUCT_DESCRIPTOR_PROPERTY)) {
      requiredProduct = targetDescriptorJsonElement.getAsJsonObject().get(REQUIRED_PRODUCT_DESCRIPTOR_PROPERTY).getAsString();
    }
    if (exists(targetDescriptorPath)) {
      JsonElement userCustomizedDescriptorJsonElement = getJsonElement(targetDescriptorPath);
      if (userCustomizedDescriptorJsonElement.getAsJsonObject().has(MIN_MULE_VERSION_DESCRIPTOR_PROPERTY)) {
        minMuleVersion =
            userCustomizedDescriptorJsonElement.getAsJsonObject().get(MIN_MULE_VERSION_DESCRIPTOR_PROPERTY).getAsString();
      }
      MuleVersion muleVersion = new MuleVersion(minMuleVersion);
      validateMuleVersion(muleVersion);
      if (userCustomizedDescriptorJsonElement.getAsJsonObject().has(REQUIRED_PRODUCT_DESCRIPTOR_PROPERTY)) {
        requiredProduct =
            userCustomizedDescriptorJsonElement.getAsJsonObject().get(REQUIRED_PRODUCT_DESCRIPTOR_PROPERTY).getAsString();
        checkState(requiredProduct.equals(MULE_CE_PRODUCT) || requiredProduct.equals(MULE_EE_PRODUCT),
                   format("%s is invalid, value is %s and valid values are (%s, %s)", REQUIRED_PRODUCT_DESCRIPTOR_PROPERTY,
                          requiredProduct, MULE_CE_PRODUCT, MULE_EE_PRODUCT));
      }
    }

    checkState(minMuleVersion != null, "Unable to resolve the Extension " + MIN_MULE_VERSION_ATTRIBUTE);
    targetDescriptorJsonElement.getAsJsonObject().addProperty(MIN_MULE_VERSION_ATTRIBUTE, minMuleVersion);
    targetDescriptorJsonElement.getAsJsonObject().addProperty(REQUIRED_PRODUCT_DESCRIPTOR_PROPERTY, requiredProduct);
    try {
      targetDescriptorPath.getParent().toFile().mkdirs();
      write(targetDescriptorPath, new GsonBuilder().setPrettyPrinting().create().toJson(targetDescriptorJsonElement).getBytes(),
            StandardOpenOption.CREATE);
    } catch (IOException e) {
      throw new MojoExecutionException(e.getMessage(), e);
    }
  }

  private void validateMuleVersion(MuleVersion muleVersion) {
    if (!muleVersion.hasSuffix()) {
      return;
    }

    checkArgument(SNAPSHOT_SUFFIX.equals(muleVersion.getSuffix()),
                  MIN_MULE_VERSION_ATTRIBUTE + " cannot have suffix different from " + SNAPSHOT_SUFFIX);

    checkArgument(project.getVersion().contains(SNAPSHOT_SUFFIX),
                  MIN_MULE_VERSION_ATTRIBUTE +
                      " declares an SNAPSHOT version, but the current project has version " + project.getVersion());
  }

  private JsonElement getJsonElement(Path targetDescriptorPath) throws MojoExecutionException {
    byte[] descriptorContent;
    try {
      descriptorContent = readAllBytes(targetDescriptorPath);
    } catch (IOException e) {
      throw new MojoExecutionException("failure opening file " + targetDescriptorPath.toFile().getAbsolutePath(), e);
    }
    JsonParser parser = new JsonParser();
    return parser.parse(new String(descriptorContent));
  }

  /**
   * It assumes there will be one, and only one,
   * <p>
   * <pre>
   * mule - fillWithAName.xml
   * </pre>
   * <p>
   * module in the current working directory.
   *
   * @param descriptorTargetPath                 final path where the descriptor must be before being either consumed or packaged. See
   *                                             {@link #descriptorPathOrFail(File)}
   * @throws MojoExecutionException if there aren't any module to read, or if the module misses the {@link #NAME_ATTRIBUTE} or
   *                                {@link #MIN_MULE_VERSION_ATTRIBUTE} attributes
   */
  private void createDescriptor(Path descriptorTargetPath)
      throws MojoExecutionException {
    final String baseDirectory = project.getBuild().getOutputDirectory();
    File moduleFile = getModuleFile(baseDirectory);
    getLog().info(format("Generating [%s] descriptor for the <module> found in [%s]", MULE_ARTIFACT_JSON,
                         moduleFile.getAbsolutePath()));

    final Document doc = getModule(moduleFile).get();
    final String name = doc.getDocumentElement().getAttribute(NAME_ATTRIBUTE);
    if (isBlank(name)) {
      throw new MojoExecutionException(format("There was an issue storing the dynamically generated descriptor file to [%s]",
                                              descriptorTargetPath));
    }

    final String relativeModuleFileName =
        moduleFile.getAbsolutePath().substring(baseDirectory.length() + 1,
                                               moduleFile.getAbsolutePath().length());
    createDescriptor(descriptorTargetPath, name, relativeModuleFileName, doc, false);
  }

  private void createDescriptor(Path descriptorPath, String name, String resourceXml, Document moduleDocument,
                                boolean validateXml)
      throws MojoExecutionException {
    final MulePluginModel.MulePluginModelBuilder mulePluginModelBuilder = new MulePluginModel.MulePluginModelBuilder()
        .setName(name);
    mulePluginModelBuilder.withBundleDescriptorLoader(new MuleArtifactLoaderDescriptor(MULE_LOADER_ID, emptyMap()));
    mulePluginModelBuilder.withClassLoaderModelDescriptorLoader(new MuleArtifactLoaderDescriptor(MULE_LOADER_ID, emptyMap()));
    mulePluginModelBuilder.withExtensionModelDescriber().setId(XML_BASED_EXTENSION_MODEL_LOADER)
        .addProperty(RESOURCE_XML, resourceXml)
        .addProperty(VALIDATE_XML, validateXml);

    Element moduleElement = moduleDocument.getDocumentElement();
    String category = moduleElement.getAttribute("category");
    String vendor = moduleElement.getAttribute("vendor");
    String requiredEntitlement = moduleElement.getAttribute("requiredEntitlement");
    String allowsEvaluationLicense = moduleElement.getAttribute("allowsEvaluationLicense");

    mulePluginModelBuilder.setRequiredProduct(isBlank(category) || "COMMUNITY".equals(category) ? MULE : MULE_EE);
    if (!isBlank(requiredEntitlement)) {
      mulePluginModelBuilder.withLicenseModel()
          .setRequiredEntitlement(requiredEntitlement)
          .setProvider(isBlank(vendor) ? "MuleSoft" : vendor)
          .setAllowsEvaluationLicense(isBlank(allowsEvaluationLicense) || "true".equals(allowsEvaluationLicense) ? true : false);
    }

    final MulePluginModel build = mulePluginModelBuilder.build();
    final String descriptor = new MulePluginModelJsonSerializer().serialize(build);
    try {
      descriptorPath.getParent().toFile().mkdirs();
      write(descriptorPath, descriptor.getBytes(), StandardOpenOption.CREATE);
    } catch (IOException e) {
      throw new MojoExecutionException(format("There was an issue storing the dynamically generated descriptor file to [%s]",
                                              descriptorPath),
                                       e);
    }
  }

  /**
   * Looks in {@code baseDirectory} recursively looking for files that start with {@link #PREFIX_SMART_CONNECTOR_NAME} and ends
   * with {@link #SUFFIX_SMART_CONNECTOR_NAME} to then validate through {@link #getModule(File)} if they are a <module/> or not.
   *
   * @param baseDirectory base path to start looking for <module/>
   * @return a {@link File} that targets the only <module/> of the current project.
   * @throws MojoExecutionException if the amount of <module/>s found is different than 1. There can be only one, Highlander.
   */
  private File getModuleFile(String baseDirectory) throws MojoExecutionException {
    final List<File> modulesFiles = listFiles(new File(baseDirectory), new FileFileFilter() {

      @Override
      public boolean accept(File file) {
        return file.getName().startsWith(PREFIX_SMART_CONNECTOR_NAME)
            && file.getName().endsWith(SUFFIX_SMART_CONNECTOR_NAME);
      }
    }, TrueFileFilter.INSTANCE)
        .stream()
        .filter(file -> getModule(file).isPresent())
        .collect(Collectors.toList());

    if (modulesFiles.size() > 1) {
      final String xmlModules = join("\n,", modulesFiles.stream().map(File::getAbsolutePath).collect(Collectors.toList()));
      throw new MojoExecutionException(format(
                                              "There are several XML files that have a <module> root element, when there must be only one. Files with <module> as root element are: [%s]",
                                              xmlModules));
    } else if (modulesFiles.isEmpty()) {
      throw new MojoExecutionException(format(
                                              "There's no XML files that has a <module> root element, thus is impossible to auto generate a [%s] descriptor file. The file must start with [%s] and end with [%s], such as [%s]",
                                              MULE_ARTIFACT_JSON,
                                              PREFIX_SMART_CONNECTOR_NAME,
                                              SUFFIX_SMART_CONNECTOR_NAME,
                                              PREFIX_SMART_CONNECTOR_NAME + "foo" + SUFFIX_SMART_CONNECTOR_NAME));
    }
    return modulesFiles.get(0);
  }

  /**
   * Given an existing {@link File}, it returns a {@link Document} if it's able to parse it and the root element's name matches
   * with {@link #MODULE_ROOT_ELEMENT}.
   *
   * @param file to be read
   * @return the parsed file if it's a <module/>
   */
  private Optional<Document> getModule(File file) {
    Optional<Document> result = empty();
    DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
    try {
      DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();
      Document doc = dBuilder.parse(file);
      if (doc.getDocumentElement().getNodeName().equals(MODULE_ROOT_ELEMENT)) {
        result = of(doc);
      }
    } catch (ParserConfigurationException | SAXException | IOException e) {
      // If it fails, then the file wasn't a <module> after all :)
    }
    return result;
  }
}
