/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extension.maven.documentation.types;

import static java.lang.String.format;
import static org.mule.extension.maven.documentation.NameUtils.prettify;
import static org.mule.metadata.api.utils.MetadataTypeUtils.getLocalPart;
import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;
import static org.mule.runtime.extension.api.util.ExtensionMetadataTypeUtils.getId;
import static org.mule.runtime.extension.api.util.ExtensionMetadataTypeUtils.isInfrastructure;
import org.mule.extension.maven.documentation.internal.model.XmlExtensionDocumentation;
import org.mule.extension.maven.documentation.internal.model.XmlExtensionElementDocumentation;
import org.mule.extension.maven.documentation.internal.model.XmlExtensionParameterDocumentation;
import org.mule.metadata.api.annotation.DefaultValueAnnotation;
import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.TypeAliasAnnotation;
import org.mule.metadata.api.model.ObjectType;

import java.util.Optional;
import java.util.StringJoiner;

class ObjectTypeUtils {

  public static final String EMPTY = "";

  static Optional<String> getObjectTypeId(ObjectType objectType) {
    Optional<String> id = getTypeId(objectType);
    Optional<String> alias = objectType.getAnnotation(TypeAliasAnnotation.class).map(TypeAliasAnnotation::getValue);
    if (id.filter(value -> value.startsWith("javax.") || value.startsWith("java.")).isPresent()) {
      return isInfrastructure(objectType) && alias.isPresent() ? alias : Optional.empty();
    }
    return alias.isPresent() ? alias : id.map(v -> v.contains(".") ? v.substring(v.lastIndexOf(".") + 1) : v);
  }

  static String toAsciidoc(ObjectType type, TypeFlattener flattener,
                           XmlExtensionDocumentation xmlDocumentation) {

    Optional<XmlExtensionElementDocumentation> typeDoc;
    if (xmlDocumentation != null && xmlDocumentation.getTypes() != null) {
      typeDoc = getId(type)
          .map(id -> xmlDocumentation.getTypes().stream().filter(doc -> doc.getName().equals(id)).findFirst().orElse(null));
    } else {
      typeDoc = Optional.empty();
    }

    StringJoiner stringJoiner = new StringJoiner("\n");
    stringJoiner.add("[cols=\".^20%,.^25%,.^30%,.^15%,.^10%\", options=\"header\"]");
    stringJoiner.add("|======================");
    stringJoiner.add("| Field | Type | Description | Default Value | Required");
    type.getFields().forEach(field -> {
      String desc = field.getAnnotation(DescriptionAnnotation.class).map(DescriptionAnnotation::getValue)
          .orElseGet(() -> {
            if (typeDoc.isPresent()) {
              return typeDoc.get().getParameters().stream()
                  .filter(p -> p.getName().equals(field.getKey().getName().getLocalPart())).findFirst()
                  .map(XmlExtensionParameterDocumentation::getDescription).orElse("");
            }
            return "";
          });
      String localPart = getLocalPart(field);
      String defaultValue = field.getAnnotation(DefaultValueAnnotation.class)
          .map(DefaultValueAnnotation::getValue)
          .orElse(EMPTY);

      stringJoiner.add(format("| %s a| %s | %s | %s | %s", prettify(localPart), flattener.flat(field.getValue()),
                              desc, defaultValue, field.isRequired() ? "x" : ""));
    });
    stringJoiner.add("|======================");
    return stringJoiner.toString();
  }

  static boolean isMessage(ObjectType objectType) {
    Optional<String> typeId = getId(objectType);
    return typeId.isPresent() && typeId.get().equals(org.mule.runtime.api.message.Message.class.getName());
  }
}
