/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extension.maven.util;

import static java.lang.String.format;
import static org.mule.plugin.maven.AbstractMuleMojo.MULE_PLUGIN_JSON_JAR_DESTINATION;
import static org.mule.plugin.maven.AbstractPackagePluginMojo.MULE_PLUGIN_CLASSIFIER;
import org.mule.extension.maven.ExtensionPackageMojo;
import org.mule.plugin.maven.AbstractMuleMojo;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.deployment.persistence.MulePluginModelJsonSerializer;

import java.io.IOException;
import java.io.InputStream;
import java.util.Optional;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.apache.commons.io.IOUtils;
import org.apache.maven.artifact.Artifact;

/**
 * Utility class to read information from {@link Artifact}s
 *
 * @since 1.3
 */
public class MulePluginArtifactLoaderUtils {

  /**
   * Returns a descriptor {@link MulePluginModel} for any given artifact if possible of reading it.
   * Scenarios where it's not possible:
   * <ol>
   *   <li>the {@code artifact} <b>is not a Mule plugin</b>, such as any Java library (e.g.: guava, apache, etc.)</li>
   *   <li>the {@code artifact} <b>is a Mule plugin</b>, but doesn't have a {@link AbstractMuleMojo#MULE_PLUGIN_JSON_JAR_DESTINATION}</li>
   * </ol>
   *
   * @param artifact that might contain the {@link ExtensionPackageMojo#MULE_PLUGIN_JSON_JAR_DESTINATION}
   * @return a {@link MulePluginModel} if exists in the current JAR file, {@link Optional#empty()} otherwise (file-commons, guava,
   * etc.)
   */
  public static Optional<MulePluginModel> readMulePluginModel(Artifact artifact) {
    Optional<MulePluginModel> result = Optional.empty();
    if (MULE_PLUGIN_CLASSIFIER.equals(artifact.getClassifier())) {
      try {
        final ZipFile pluginJar = new ZipFile(artifact.getFile());
        final ZipEntry mulePluginDescriptorEntry = pluginJar.getEntry(toZipPath(MULE_PLUGIN_JSON_JAR_DESTINATION));
        if (mulePluginDescriptorEntry != null) {
          try (InputStream stream = pluginJar.getInputStream(mulePluginDescriptorEntry)) {
            result = Optional.of(new MulePluginModelJsonSerializer().deserialize(IOUtils.toString(stream)));
          } catch (IOException e) {
            throw new IllegalArgumentException(format("Cannot get the file [%s] within the [%s] JAR for the following artifact [%s]",
                                                      MULE_PLUGIN_JSON_JAR_DESTINATION, artifact.getFile().getAbsolutePath(),
                                                      artifact.toString()),
                                               e);
          }
        }
      } catch (IOException e) {
        throw new IllegalArgumentException(format("Cannot get the file [%s] JAR for the following artifact [%s]",
                                                  artifact.getFile().getAbsolutePath(), artifact.toString()),
                                           e);
      }
    }
    return result;
  }

  public static String toZipPath(String filePath) {
    return filePath.replace("\\", "/");
  }

}
