package org.mule.extension.maven;

import static java.lang.Boolean.parseBoolean;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.maven.artifact.ArtifactUtils.versionlessKey;
import static org.apache.maven.plugins.annotations.LifecyclePhase.PACKAGE;
import static org.apache.maven.plugins.annotations.ResolutionScope.TEST;
import static org.mule.extension.maven.loader.MulePluginJsonDescriberLoader.loadMulePluginDescriber;
import static org.mule.runtime.api.util.MuleSystemProperties.FORCE_EXTENSION_VALIDATION_PROPERTY_NAME;
import org.mule.extension.maven.loader.MavenProjectExtensionModelLoader;
import org.mule.plugin.maven.AbstractMuleMojo;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.extension.api.persistence.ExtensionModelJsonSerializer;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.net.MalformedURLException;
import java.util.List;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.ArtifactUtils;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.descriptor.PluginDescriptor;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProjectHelper;
import org.apache.maven.project.ProjectDependenciesResolver;

/**
 * This Maven Plugin takes the generated resource {@link ExtensionPackageMojo#MULE_ARTIFACT_JSON}, to look for the ID on it so that
 * through SPI can generate the {@link ExtensionModel} properly. To do so, the invoker of the plugin must have the required
 * dependencies in its classpath.
 * <p/>
 * Once it generates the {@link ExtensionModel} in memory, it serializes it in and then adds to the current project. If we take
 * the HTTP plugin as sample, the generated JAR artifact will be:
 *
 * <pre>
 * /Users/someUser/.m2/repository/org/mule/modules/mule-module-http-ext/4.0-SNAPSHOT/mule-module-http-ext-4.0-SNAPSHOT-mule-plugin.jar
 * </pre>
 *
 * and the generated serialized JSON {@link ExtensionModel} will be:
 *
 * <pre>
 * /Users/lautar0/.m2/repository/org/mule/modules/mule-module-http-ext/4.0-SNAPSHOT/mule-module-http-ext-4.0-SNAPSHOT-extension-model-4.0.0.json
 * </pre>
 * <p/>
 * This {@link Mojo} must be executed on Mule Plugins that generate {@link ExtensionModel}.
 *
 * @since 1.0
 */
@Mojo(name = "extension-model", defaultPhase = PACKAGE, requiresDependencyResolution = TEST, threadSafe = true)
public class ExtensionModelMojo extends AbstractMuleMojo {

  public static final String TEMPORAL_EXTENSION_MODEL_JSON = "temporal-extension-model.json";
  public static final String SKIP_EXTENSION_MODEL_VALIDATION = "skipExtensionModelValidation";

  private static final String ORG_MULE_RUNTIME = "org.mule.runtime";
  private static final String COM_MULESOFT_MULE_RUNTIME_MODULES = "com.mulesoft.mule.runtime.modules";

  @Parameter(defaultValue = "${session}", readonly = true, required = true)
  private MavenSession session;

  @Component
  private ProjectDependenciesResolver dependenciesResolver;

  @Component
  private MavenProjectHelper projectHelper;

  @Parameter(defaultValue = "${plugin}", readonly = true)
  private PluginDescriptor pluginDescriptor;

  /**
   * Whether the generation of the extension model should force validation. By default it validates it.
   */
  @Parameter(defaultValue = "false", property = SKIP_EXTENSION_MODEL_VALIDATION)
  private Boolean skipExtensionModelValidation;

  @Override
  public void execute() throws MojoExecutionException, MojoFailureException {
    if (parseBoolean(System.getProperty("mule.maven.extension.model.disable", "false"))) {
      getLog().info("Extension Model generation skipped.");
      return;
    }

    if (skipExtensionModelValidation) {
      getLog()
          .warn("Extension Model generation will be done by omitting validations, thus it wil not validate it's syntactically correct (BE SURE TO TEST THE CONNECTOR).");
    }
    System.setProperty(FORCE_EXTENSION_VALIDATION_PROPERTY_NAME, String.valueOf(skipExtensionModelValidation));

    MulePluginModel mulePluginDescriber = loadMulePluginDescriber(outputDirectory);

    MavenProjectExtensionModelLoader extensionModelLoader = new MavenProjectExtensionModelLoader(getLog());
    enrichPluginClassLoader();
    final ExtensionModel extensionModel = extensionModelLoader.loadExtension(project, session);
    final String serializedExtensionModel = new ExtensionModelJsonSerializer(true).serialize(extensionModel);
    final File generatedExtensionModelFile = new File(outputDirectory, TEMPORAL_EXTENSION_MODEL_JSON);
    try {
      try (PrintWriter out = new PrintWriter(generatedExtensionModelFile)) {
        out.println(serializedExtensionModel);
      }
    } catch (FileNotFoundException e) {
      throw new MojoFailureException(
                                     format("Failure while saving the serialized ExtensionModel to the file [%s]",
                                            generatedExtensionModelFile.getAbsolutePath()));
    }
    projectHelper.attachArtifact(project, "json", "extension-model-" + mulePluginDescriber.getMinMuleVersion(),
                                 generatedExtensionModelFile);
  }

  private void enrichPluginClassLoader() throws MojoExecutionException {
    List<String> pluginRuntimeLibraries = pluginDescriptor.getArtifacts().stream()
        .filter(artifact -> artifact.getGroupId().equals(ORG_MULE_RUNTIME)
            || artifact.getGroupId().equals(COM_MULESOFT_MULE_RUNTIME_MODULES))
        .map(ArtifactUtils::versionlessKey)
        .collect(toList());

    List<Artifact> runtimeProvidedLibraries = project.getArtifacts().stream()
        .filter(artifact -> artifact.getScope().equals("provided"))
        .filter(artifact -> artifact.getGroupId().equals(ORG_MULE_RUNTIME)
            || artifact.getGroupId().equals(COM_MULESOFT_MULE_RUNTIME_MODULES))
        .filter(artifact -> artifact.getType().equals("jar"))
        .filter(artifact -> !pluginDescriptor.getArtifacts().contains(artifact))
        .filter(artifact -> !pluginRuntimeLibraries.contains(versionlessKey(artifact)))
        .collect(toList());

    for (Artifact runtimeProvidedLibrary : runtimeProvidedLibraries) {
      getLog().info(format("Adding URL to plugin class loader for provided Mule Runtime library: %s", runtimeProvidedLibrary));
      try {
        pluginDescriptor.getClassRealm().addURL(runtimeProvidedLibrary.getFile().toURI().toURL());
      } catch (MalformedURLException e) {
        throw new MojoExecutionException("Error while getting Mule Runtime provided library from project", e);
      }

    }
  }

}
