/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.extension.maven.loader;

import static java.lang.String.format;
import static java.lang.String.join;
import static java.util.stream.Collectors.toCollection;
import static org.apache.maven.artifact.Artifact.SCOPE_TEST;
import static org.mule.extension.maven.ExtensionModelMojo.SKIP_EXTENSION_MODEL_VALIDATION;
import static org.mule.plugin.maven.AbstractPackagePluginMojo.MULE_PLUGIN_CLASSIFIER;
import static org.mule.runtime.api.dsl.DslResolvingContext.getDefault;

import org.mule.extension.maven.util.MulePluginArtifactLoaderUtils;
import org.mule.plugin.maven.AbstractPackagePluginMojo;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptor;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.extension.api.loader.ExtensionModelLoader;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.Set;

import org.apache.commons.lang3.tuple.Pair;
import org.apache.maven.artifact.DependencyResolutionRequiredException;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.DefaultDependencyResolutionRequest;
import org.apache.maven.project.DependencyResolutionException;
import org.apache.maven.project.DependencyResolutionResult;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.ProjectDependenciesResolver;
import org.eclipse.aether.graph.DependencyNode;
import org.eclipse.aether.util.filter.AndDependencyFilter;
import org.eclipse.aether.util.filter.ScopeDependencyFilter;

/**
 * Loader of an {@link ExtensionModel} for a Mule plugin artifact from an extension {@link MavenProject}.
 *
 * @since 1.0
 */
public class MavenProjectExtensionModelLoader {

  private final Log log;

  public MavenProjectExtensionModelLoader(Log log) {
    this.log = log;
  }

  /**
   * Builds an {@link ExtensionModel} from a {@link MavenProject} and a {@link MulePluginModel} that holds the information about
   * the extension mule plugin that is wanted to be loaded.
   *
   * @param shouldValidateExtensionModel when set to true, all mule-plugins the current build relies on will be used to
   *                                     generate a set of ExtensionModels to be used later on for the current mule connector
   *                                     being built.
   *                                     If set to false, NO mule-plugin dependency will generate an extension model leaving the mentioned set empty.
   * @return an {@link ExtensionModel} built up from the extension {@link MavenProject} provided.
   * @throws MojoFailureException
   */
  public ExtensionModel loadExtension(MavenProject project, MavenSession session,
                                      ProjectDependenciesResolver dependenciesResolver, MulePluginModel mulePluginDescriber,
                                      boolean shouldValidateExtensionModel)
      throws MojoFailureException, MojoExecutionException {
    final ClassLoader pluginClassLoader = getInvokerClassLoader(project);
    final Set<ExtensionModel> extensions;
    if (shouldValidateExtensionModel) {
      extensions = getPluginsExtensions(project, session, dependenciesResolver, pluginClassLoader);
    } else {
      extensions = new HashSet<>();
    }
    return getExtensionModel(pluginClassLoader, extensions, mulePluginDescriber);
  }

  /**
   * Goes over all the plugin dependencies generating the needed {@link ExtensionModel} for the current project to work properly.
   *
   * @param project to look for dependencies
   * @param session to do dependency resolution
   * @param dependenciesResolver to do dependency resolution
   * @param pluginClassLoader class loader with all the needed JARs to load the {@link ExtensionModel}s
   * @return a set of {@link ExtensionModel} of the current plugin's dependencies
   * @throws MojoFailureException if there are plugins that could not generate an {@link ExtensionModel} despite they have a
   *         {@link MulePluginModel#getExtensionModelLoaderDescriptor()}
   * @throws MojoExecutionException if there are plugins that could not generate an {@link ExtensionModel} despite they have a
   *         {@link MulePluginModel#getExtensionModelLoaderDescriptor()}
   */
  private Set<ExtensionModel> getPluginsExtensions(MavenProject project, MavenSession session,
                                                   ProjectDependenciesResolver dependenciesResolver,
                                                   ClassLoader pluginClassLoader)
      throws MojoFailureException, MojoExecutionException {
    final Set<ExtensionModel> extensions = new LinkedHashSet<>();

    for (MulePluginModel mulePluginDescriber : filterPluginsWithExtensionModelLoader(project, session, dependenciesResolver)) {
      try {
        ExtensionModel extensionModel = getExtensionModel(pluginClassLoader, extensions, mulePluginDescriber);
        extensions.add(extensionModel);
      } catch (RuntimeException e) {
        String message =
            format("There was an issue trying to create ExtensionModel for the dependency mule-plugin:[%s] (this might be caused due to classloading issues). "
                +
                "If you want to disable strict validations for your build, parameterize -D%s=true (be sure to test your connector to be sure there won't be syntactic errors)",
                   mulePluginDescriber.getName(), SKIP_EXTENSION_MODEL_VALIDATION);
        log.warn(message, e);
        throw e;
      }
    }

    return extensions;
  }

  /**
   * Goes over all the dependencies (see {@link MavenProject#getArtifacts()}) that are
   * {@link AbstractPackagePluginMojo#MULE_PLUGIN_CLASSIFIER} while filter them by having a mule-artifact.json descriptor, as
   * there are {@link AbstractPackagePluginMojo#MULE_PLUGIN_CLASSIFIER} that do not have the specific descriptor (such as
   * file-common) by lastly checking they have an {@link MulePluginModel#getExtensionModelLoaderDescriptor()}
   *
   * @param project to look for all the dependencies
   * @param session to do dependency resolution
   * @param dependenciesResolver to do dependency resolution
   * @return a set of {@link MulePluginModel}, not null.
   */
  private Set<MulePluginModel> filterPluginsWithExtensionModelLoader(MavenProject project, MavenSession session,
                                                                     ProjectDependenciesResolver dependenciesResolver)
      throws MojoExecutionException {
    DefaultDependencyResolutionRequest request = new DefaultDependencyResolutionRequest(project, session.getRepositorySession());
    request.setResolutionFilter(AndDependencyFilter.newInstance(new ScopeDependencyFilter(SCOPE_TEST),
                                                                (node, parents) -> MULE_PLUGIN_CLASSIFIER
                                                                    .equals(node.getArtifact().getClassifier())));

    Set<Pair<String, String>> dependenciesByGroupArtifactId = new HashSet<>();
    try {
      final DependencyResolutionResult result = dependenciesResolver.resolve(request);

      List<DependencyNode> seenNodes = new ArrayList<>();
      seenNodes.add(result.getDependencyGraph());

      populateDependenciesTransitively(dependenciesByGroupArtifactId, result.getDependencyGraph().getChildren(), seenNodes);
    } catch (DependencyResolutionException e) {
      throw new MojoExecutionException("Error building dependency graph.", e);
    }

    return project.getArtifacts().stream()
        .filter(artifact -> !SCOPE_TEST.equals(artifact.getScope()))
        .sorted((o1, o2) -> {
          if (dependenciesByGroupArtifactId
              .contains(Pair.of(o1.getGroupId() + ":" + o1.getArtifactId(), o2.getGroupId() + ":" + o2.getArtifactId()))) {
            return 1;
          } else if (dependenciesByGroupArtifactId
              .contains(Pair.of(o2.getGroupId() + ":" + o2.getArtifactId(), o1.getGroupId() + ":" + o1.getArtifactId()))) {
            return -1;
          } else {
            return 0;
          }
        })
        .map(MulePluginArtifactLoaderUtils::readMulePluginModel)
        .filter(Optional::isPresent)
        .map(Optional::get)
        .filter(mulePluginModel -> mulePluginModel.getExtensionModelLoaderDescriptor().isPresent())
        .collect(toCollection(LinkedHashSet::new));
  }

  private void populateDependenciesTransitively(Set<Pair<String, String>> dependenciesByGroupArtifactId,
                                                final List<DependencyNode> children,
                                                List<DependencyNode> seenNodes) {
    for (DependencyNode depNode : children) {
      if (!SCOPE_TEST.equals(depNode.getDependency().getScope())
          && MULE_PLUGIN_CLASSIFIER.equals(depNode.getArtifact().getClassifier())) {

        final List<DependencyNode> localSeenNodes = new ArrayList<>(seenNodes);
        localSeenNodes.add(depNode);

        for (DependencyNode localSeenNode : localSeenNodes) {
          final String localSeenNodeId =
              localSeenNode.getArtifact().getGroupId() + ":" + localSeenNode.getArtifact().getArtifactId();
          final String depNodeId =
              depNode.getArtifact().getGroupId() + ":" + depNode.getArtifact().getArtifactId();

          if (!localSeenNodeId.equals(depNodeId)) {
            dependenciesByGroupArtifactId
                .add(Pair.of(localSeenNodeId, depNodeId));
          }
        }

        populateDependenciesTransitively(dependenciesByGroupArtifactId, depNode.getChildren(), localSeenNodes);
      }
    }
  }

  private ExtensionModel getExtensionModel(ClassLoader pluginClassLoader, Set<ExtensionModel> extensions,
                                           MulePluginModel mulePluginDescriber)
      throws MojoExecutionException, MojoFailureException {
    MuleArtifactLoaderDescriptor loaderDescriptor = getExtensionLoaderDescriptor(mulePluginDescriber);
    log.debug(format("Creating ExtensionModel for name:[%s], ID:[%s]", mulePluginDescriber.getName(), loaderDescriptor.getId()));
    final ExtensionModelLoader extensionModelLoader = getExtensionModelLoader(loaderDescriptor.getId(), pluginClassLoader);
    ExtensionModel extensionModel =
        extensionModelLoader.loadExtensionModel(pluginClassLoader, getDefault(extensions), loaderDescriptor.getAttributes());
    return extensionModel;
  }

  /**
   * @param jsonDescriber an {@link MulePluginModel} instance holding the mule plugin information.
   * @return an {@link MuleArtifactLoaderDescriptor} instance.
   */
  private MuleArtifactLoaderDescriptor getExtensionLoaderDescriptor(MulePluginModel jsonDescriber)
      throws MojoExecutionException, MojoFailureException {
    return jsonDescriber.getExtensionModelLoaderDescriptor()
        .orElseThrow(() -> new MojoExecutionException(
                                                      format("The plugin [%s] does not have a ExtensionLoader descriptor, nothing to generate so far.",
                                                             jsonDescriber.getName())));
  }

  /**
   * Assembles a {@link ClassLoader} from the classpath of the invoker of the current maven's plugin (aka: the connector). This
   * will allow us to consume any {@link ExtensionModelLoader} from it's dependencies.
   *
   * @return a {@link ClassLoader} with the needed elements to find the {@link ExtensionModelLoader} implementations and then load
   *         the {@link ExtensionModel} properly.
   * @throws MojoFailureException if there are issues while looking for the classpath context or trying to generate the URLs form
   *         each item of the classpath.
   */
  private ClassLoader getInvokerClassLoader(MavenProject project) throws MojoFailureException {
    List<String> invokerClasspathElements;
    try {
      invokerClasspathElements = project.getCompileClasspathElements();
    } catch (DependencyResolutionRequiredException e) {
      throw new MojoFailureException("There was an issue trying to consume the classpath for the current project.", e);
    }
    invokerClasspathElements.add(project.getBuild().getOutputDirectory());
    if (log.isDebugEnabled()) {
      log.debug(format("Classpath to process: [%s]", join(",", invokerClasspathElements)));
    }

    URL urls[] = new URL[invokerClasspathElements.size()];
    for (int i = 0; i < invokerClasspathElements.size(); ++i) {
      final String invokerClasspathElement = invokerClasspathElements.get(i);
      try {
        urls[i] = new File(invokerClasspathElement).toURI().toURL();
      } catch (MalformedURLException e) {
        throw new MojoFailureException(
                                       format("There was an issue trying to convert the element [%s] to an URL. Full classpath: [%s]",
                                              invokerClasspathElement, join(",", invokerClasspathElements)));
      }
    }
    return new URLClassLoader(urls, getClass().getClassLoader());
  }

  /**
   * Given an {@code id} it will look for the first {@link ExtensionModelLoader} implementation, where the acceptance criteria is
   * {@link ExtensionModelLoader#getId()} being equals to it.
   *
   * @param id the ID to look for in each implementation.
   * @param pluginClassLoader {@link ClassLoader} of the current invoker.
   * @return the {@link ExtensionModelLoader} that matches to {@code id}.
   * @throws MojoFailureException if there's no {@link ExtensionModelLoader} that matches to the parametrized {@code id}.
   */
  private ExtensionModelLoader getExtensionModelLoader(String id, ClassLoader pluginClassLoader)
      throws MojoFailureException {
    List<String> foundIds = new ArrayList<>();
    final ServiceLoader<ExtensionModelLoader> extensionModelLoaders =
        ServiceLoader.load(ExtensionModelLoader.class, pluginClassLoader);
    for (ExtensionModelLoader extensionModelLoader : extensionModelLoaders) {
      foundIds.add(format("Class:[%s]; ID:[%s].", extensionModelLoader.getClass().getName(), extensionModelLoader.getId()));
      if (id.equals(extensionModelLoader.getId())) {
        return extensionModelLoader;
      }
    }
    throw new MojoFailureException(
                                   format("Failure while looking for an implementation class of [%s] class through SPI for the ID [%s]. Found resources: \n%s",
                                          ExtensionModelLoader.class.getName(), id,
                                          join(", \n", foundIds)));
  }
}
