/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.extension.maven;

import static com.google.common.base.Preconditions.checkArgument;
import static java.io.File.separator;
import static java.lang.Boolean.parseBoolean;
import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.nio.file.Files.delete;
import static java.nio.file.Files.exists;
import static java.nio.file.Files.isRegularFile;
import static java.nio.file.Files.notExists;
import static java.nio.file.Files.readAllBytes;
import static java.nio.file.Files.write;
import static java.util.Collections.emptyMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.commons.io.FileUtils.listFiles;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.uncapitalize;
import static org.apache.maven.plugins.annotations.LifecyclePhase.COMPILE;
import static org.mule.extension.maven.loader.MulePluginJsonDescriberLoader.loadMulePluginDescriber;
import static org.mule.extension.maven.util.MulePluginArtifactLoaderUtils.readMulePluginModel;
import static org.mule.plugin.maven.AbstractPackagePluginMojo.MULE_PLUGIN_CLASSIFIER;
import static org.mule.runtime.api.deployment.meta.AbstractMuleArtifactModel.MIN_MULE_VERSION;
import static org.mule.runtime.api.deployment.meta.AbstractMuleArtifactModel.REQUIRED_PRODUCT;
import static org.mule.runtime.api.deployment.meta.Product.MULE;
import static org.mule.runtime.api.deployment.meta.Product.MULE_EE;
import static org.mule.runtime.api.meta.Category.CERTIFIED;
import static org.mule.runtime.api.meta.Category.COMMUNITY;
import static org.mule.runtime.api.meta.Category.PREMIUM;
import static org.mule.runtime.api.meta.Category.SELECT;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.runtime.extension.api.annotation.Extension.MULESOFT;
import static org.mule.runtime.extension.api.util.XmlModelUtils.createXmlLanguageModel;

import org.mule.extension.maven.util.MulePluginArtifactLoaderUtils;
import org.mule.plugin.maven.AbstractMuleMojo;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptor;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptorBuilder;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.deployment.meta.Product;
import org.mule.runtime.api.deployment.persistence.MulePluginModelJsonSerializer;
import org.mule.runtime.api.meta.Category;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.model.XmlDslModel;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import org.apache.commons.io.filefilter.FileFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.model.Resource;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

/**
 * Mojo responsible of looking for an existing {@link ExtensionPackageMojo#MULE_ARTIFACT_JSON} in META-INF folder.
 * <p/>
 * If it doesn't exists, it assumes it's a <module/> scenario, to which it will try to generate one with the following
 * conventions:
 * <ol>
 * <li>Will scan the /classes looking for anything that starts with {@link #PREFIX_SMART_CONNECTOR_NAME} and ends with
 * {@link #SUFFIX_SMART_CONNECTOR_NAME}</li>
 * <li>For each element found, it will open it and look if they are valid XML files and their root elements matches to
 * {@link #MODULE_ROOT_ELEMENT}</li>
 * <li>There must be one, and only one, file in the /classes that match the previous rules. If not, fails.</li>
 * </ol>
 *
 * This mojo relies in the {@link Mojo#requiresDependencyResolution()} to be {@link ResolutionScope#COMPILE} to be sure all
 * dependencies to work with are not {@link Artifact#SCOPE_TEST}, refer to {@link #calculateProduct(String, String)}
 * @since 1.0
 */
@Mojo(name = "extension-descriptor", defaultPhase = COMPILE, threadSafe = true,
    requiresDependencyResolution = ResolutionScope.COMPILE)
public class ExtensionDescriptorMojo extends AbstractMuleMojo {

  private static final String META_INF = "META-INF";
  private static final String AUTO_GENERATED_MULE_ARTIFACT_DESCRIPTOR =
      Paths.get(META_INF, "auto-generated-" + MULE_ARTIFACT_JSON).toString();

  private static final String MULE_LOADER_ID = "mule";
  public static final String XML_BASED_EXTENSION_MODEL_LOADER = "xml-based";
  public static final String RESOURCE_XML = "resource-xml";
  public static final String VALIDATE_XML = "validate-xml";
  public static final String RESOURCES_PATHS = "resources-paths";
  private static final String PREFIX_SMART_CONNECTOR_NAME = "module-";
  private static final String SUFFIX_SMART_CONNECTOR_NAME = ".xml";
  private static final String MODULE_ROOT_ELEMENT = "module";
  private static final String NAME_ATTRIBUTE = "name";
  private static final String PREFIX_ATTRIBUTE = "prefix";
  private static final String CATEGORY_ATTRIBUTE = "category";
  private static final String MULE_VERSION_PROPERTY = "mule.version";
  private static final String MIN_MULE_VERSION_PROPERTY = "min.mule.version";
  private static final String SKIP_MULE_VERSION_SUFFIX_VALIDATION_PROPERTY = "mule.extensions.disableMuleVersionSuffixValidation";
  public static final String SNAPSHOT_SUFFIX = "SNAPSHOT";
  public static final String INTERNAL_FOLDER = "internal";
  public static final String EE_MULESOFT_ARTIFACTS_GROUP_ID_PREFIX = "com.mulesoft";

  /**
   * Returns a {@link Path} to an existing descriptor file or fails.
   *
   * @param outputDirectory to look for the current output directory
   * @return an existing {@link Path} to a descriptor file
   * @throws MojoFailureException if the descriptor file is absent (probably because this Mojo hasn't been executed)
   */
  public static Path descriptorPathOrFail(File outputDirectory) throws MojoFailureException {
    return descriptorPath(outputDirectory, "classes" + separator + ExtensionPackageMojo.MULE_PLUGIN_JSON_JAR_DESTINATION);
  }

  /**
   * Returns a {@link Path} to an auto generated descriptor.
   *
   * @param outputDirectory to look for the current output directory
   * @return an existing {@link Path} to a descriptor
   */
  public static Path autoGeneratedDescriptorPath(File outputDirectory) throws MojoFailureException {
    return Paths.get(outputDirectory.getAbsolutePath(), AUTO_GENERATED_MULE_ARTIFACT_DESCRIPTOR);
  }

  private static Path descriptorPath(File outputDirectory, String descriptorLocation) throws MojoFailureException {
    final Path path = Paths.get(outputDirectory.getAbsolutePath(), descriptorLocation);
    if (notExists(path)) {
      throw new MojoFailureException(format("Should not have reach this point, could not obtain descriptor file from [%s]",
                                            path));
    }
    return path;
  }

  @Override
  public void execute() throws MojoExecutionException, MojoFailureException {
    // This file may be created by MulePluginDescriptorGenerator
    final Path autoGeneratedDescriptorPath = autoGeneratedDescriptorPath(new File(project.getBuild().getOutputDirectory()));
    final Path targetDescriptorPath =
        Paths.get(project.getBuild().getOutputDirectory(), META_INF, MULE_ARTIFACT, MULE_ARTIFACT_JSON);
    if (!exists(autoGeneratedDescriptorPath) && !exists(targetDescriptorPath)) {
      getLog().debug(format("No [%s] descriptor found, trying to create one", targetDescriptorPath));
      try {
        createDescriptor(autoGeneratedDescriptorPath);
      } catch (MojoExecutionException e) {
        if (!exists(targetDescriptorPath)) {
          // continue if there's a manual defined descriptor.
          throw e;
        }
      }
    }
    applyUserCustomizedPropertiesAndDefaultValues(targetDescriptorPath, autoGeneratedDescriptorPath);
    MulePluginModel mulePluginModel = loadMulePluginDescriber(outputDirectory);
    validate(mulePluginModel);
    if (exists(autoGeneratedDescriptorPath)) {
      deleteQuietly(autoGeneratedDescriptorPath.toFile());
    }
  }

  private void validate(MulePluginModel mulePluginModel) {
    mulePluginModel.getLicense().ifPresent(licenseModel -> {
      licenseModel.getRequiredEntitlement().ifPresent(requiredEntitlement -> {
        String provider = licenseModel.getProvider();
        if (!MULESOFT.equals(provider)) {
          String outputDirectory = project.getBuild().getOutputDirectory();
          String keyFileName = String.format("%s-%s.key", uncapitalize(provider.replace(" ", "")), mulePluginModel.getName());
          if (!new File(outputDirectory, keyFileName).exists()) {
            throw new RuntimeException(format(
                                              "The plugin requires an entitlement but there is no key file (%s) found. The provider key file for customer licenses must be added to the classpath at the root level",
                                              keyFileName));
          }
        }
      });
    });
  }

  private void applyUserCustomizedPropertiesAndDefaultValues(Path targetDescriptorPath, Path autoGeneratedDescriptorPath)
      throws MojoExecutionException {
    String minMuleVersion = resolveMinMuleVersion();
    String requiredProduct = MULE.name();

    JsonElement targetDescriptorJsonElement;
    if (exists(autoGeneratedDescriptorPath)) {
      targetDescriptorJsonElement = getJsonElement(autoGeneratedDescriptorPath);
    } else {
      targetDescriptorJsonElement = getJsonElement(targetDescriptorPath);
    }

    if (targetDescriptorJsonElement.getAsJsonObject().has(REQUIRED_PRODUCT)) {
      requiredProduct = targetDescriptorJsonElement.getAsJsonObject().get(REQUIRED_PRODUCT).getAsString();
    }
    if (exists(targetDescriptorPath)) {
      JsonElement userCustomizedDescriptorJsonElement = getJsonElement(targetDescriptorPath);
      if (userCustomizedDescriptorJsonElement.getAsJsonObject().has(MIN_MULE_VERSION)) {
        minMuleVersion =
            userCustomizedDescriptorJsonElement.getAsJsonObject().get(MIN_MULE_VERSION).getAsString();
      }
      MuleVersion muleVersion = new MuleVersion(minMuleVersion);
      validateMuleVersion(muleVersion);
      if (userCustomizedDescriptorJsonElement.getAsJsonObject().has(REQUIRED_PRODUCT)) {
        requiredProduct =
            userCustomizedDescriptorJsonElement.getAsJsonObject().get(REQUIRED_PRODUCT).getAsString();
        checkState(requiredProduct.equals(MULE.name()) || requiredProduct.equals(MULE_EE.name()),
                   format("%s is invalid, value is %s and valid values are (%s, %s)", REQUIRED_PRODUCT,
                          requiredProduct, MULE.name(), MULE_EE.name()));
      }
    }

    checkState(minMuleVersion != null, "Unable to resolve the Extension " + MIN_MULE_VERSION);
    targetDescriptorJsonElement.getAsJsonObject().addProperty(MIN_MULE_VERSION, minMuleVersion);
    targetDescriptorJsonElement.getAsJsonObject().addProperty(REQUIRED_PRODUCT, requiredProduct);
    try {
      targetDescriptorPath.getParent().toFile().mkdirs();
      if (exists(targetDescriptorPath)) {
        delete(targetDescriptorPath);
      }
      write(targetDescriptorPath, new GsonBuilder().setPrettyPrinting().create().toJson(targetDescriptorJsonElement).getBytes(),
            StandardOpenOption.CREATE);
    } catch (IOException e) {
      throw new MojoExecutionException(e.getMessage(), e);
    }
  }

  private String resolveMinMuleVersion() {
    String muleVersion = (String) project.getProperties().get(MULE_VERSION_PROPERTY);
    Object minMuleVersion = project.getProperties().get(MIN_MULE_VERSION_PROPERTY);

    if (minMuleVersion == null) {
      return muleVersion;
    }

    MuleVersion muleVersionObject = new MuleVersion(muleVersion);
    MuleVersion minMuleVersionObject = new MuleVersion((String) minMuleVersion);

    checkArgument(minMuleVersionObject.atLeast(muleVersionObject), "Min mule version chosen " +
        "(" + minMuleVersionObject + ") has to be at least equal or higher than the mule version (" + muleVersionObject + ")");

    return (String) minMuleVersion;
  }

  private void validateMuleVersion(MuleVersion muleVersion) {
    if (!muleVersion.hasSuffix() || parseBoolean(getProperty(SKIP_MULE_VERSION_SUFFIX_VALIDATION_PROPERTY, "false"))) {
      return;
    }

    checkArgument(SNAPSHOT_SUFFIX.equals(muleVersion.getSuffix()),
                  MIN_MULE_VERSION + " cannot have suffix different from " + SNAPSHOT_SUFFIX);

    checkArgument(project.getVersion().contains(SNAPSHOT_SUFFIX),
                  MIN_MULE_VERSION +
                      " declares an SNAPSHOT version, but the current project has version " + project.getVersion());
  }

  private JsonElement getJsonElement(Path targetDescriptorPath) throws MojoExecutionException {
    byte[] descriptorContent;
    try {
      descriptorContent = readAllBytes(targetDescriptorPath);
    } catch (IOException e) {
      throw new MojoExecutionException("failure opening file " + targetDescriptorPath.toFile().getAbsolutePath(), e);
    }
    JsonParser parser = new JsonParser();
    return parser.parse(new String(descriptorContent));
  }

  /**
   * It assumes there will be one, and only one,
   * <p>
   * <pre>
   * mule - fillWithAName.xml
   * </pre>
   * <p>
   * module in the current working directory.
   *
   * @param descriptorTargetPath                 final path where the descriptor must be before being either consumed or packaged. See
   *                                             {@link #descriptorPathOrFail(File)}
   * @throws MojoExecutionException if there aren't any module to read, or if the module misses the {@link #NAME_ATTRIBUTE}
   */
  private void createDescriptor(Path descriptorTargetPath)
      throws MojoExecutionException {
    final String baseDirectory = project.getBuild().getOutputDirectory();
    File moduleFile = getModuleFile(baseDirectory);
    getLog().info(format("Generating [%s] descriptor for the <module> found in [%s]", MULE_ARTIFACT_JSON,
                         moduleFile.getAbsolutePath()));

    final Document doc = getModule(moduleFile).get();
    final String name = doc.getDocumentElement().getAttribute(NAME_ATTRIBUTE);
    if (isBlank(name)) {
      throw new MojoExecutionException(format("There was an issue storing the dynamically generated descriptor file to [%s]",
                                              descriptorTargetPath));
    }

    final String relativeModuleFileName =
        moduleFile.toURI().getPath().substring(new File(baseDirectory).toURI().getPath().length());
    createDescriptor(descriptorTargetPath, name, relativeModuleFileName, doc);
  }

  private void createDescriptor(Path descriptorPath, String name, String resourceXml, Document moduleDocument)
      throws MojoExecutionException {
    final MulePluginModel.MulePluginModelBuilder mulePluginModelBuilder = new MulePluginModel.MulePluginModelBuilder()
        .setName(name);
    mulePluginModelBuilder.withBundleDescriptorLoader(new MuleArtifactLoaderDescriptor(MULE_LOADER_ID, emptyMap()));

    final MuleArtifactLoaderDescriptorBuilder muleArtifactLoaderDescriptorBuilder =
        new MuleArtifactLoaderDescriptorBuilder().setId(MULE_LOADER_ID);

    final Set<String> exportedResources = calculateExportedResources(name, moduleDocument);
    if (!exportedResources.isEmpty()) {
      muleArtifactLoaderDescriptorBuilder.addProperty("exportedResources", exportedResources);
    }
    mulePluginModelBuilder.withClassLoaderModelDescriptorLoader(muleArtifactLoaderDescriptorBuilder.build());
    mulePluginModelBuilder.withExtensionModelDescriber().setId(XML_BASED_EXTENSION_MODEL_LOADER)
        .addProperty(RESOURCE_XML, resourceXml)
        .addProperty(VALIDATE_XML, false)
        .addProperty(RESOURCES_PATHS, exportedResources);

    Element moduleElement = moduleDocument.getDocumentElement();
    String vendor = moduleElement.getAttribute("vendor");
    String requiredEntitlement = moduleElement.getAttribute("requiredEntitlement");
    String allowsEvaluationLicense = moduleElement.getAttribute("allowsEvaluationLicense");

    String category = moduleElement.getAttribute(CATEGORY_ATTRIBUTE);
    mulePluginModelBuilder.setRequiredProduct(calculateProduct(name, category));

    if (!isBlank(requiredEntitlement)) {
      mulePluginModelBuilder.withLicenseModel()
          .setRequiredEntitlement(requiredEntitlement)
          .setProvider(isBlank(vendor) ? MULESOFT : vendor)
          .setAllowsEvaluationLicense(isBlank(allowsEvaluationLicense) || "true".equals(allowsEvaluationLicense));
    }

    final MulePluginModel build = mulePluginModelBuilder.build();
    final String descriptor = new MulePluginModelJsonSerializer().serialize(build);
    try {
      descriptorPath.getParent().toFile().mkdirs();
      write(descriptorPath, descriptor.getBytes(), StandardOpenOption.CREATE);
    } catch (IOException e) {
      throw new MojoExecutionException(format("There was an issue storing the dynamically generated descriptor file to [%s]",
                                              descriptorPath),
                                       e);
    }
  }

  /**
   * This method heavily relies in that all the dependencies of the current {@link MavenProject} are not {@link Artifact#SCOPE_TEST}.
   * 
   * @param name name of the current <module/>
   * @param category category of the current module, if blank/null it will assume it's {@link Category#COMMUNITY} (default value
   *                 for the <module/> XSD)
   * @return the correct {@link Product} for the current plugin being built
   * @throws MojoExecutionException if there are inconsistencies between what the module declares and what really is, e.g.: the
   * <module/> states it's {@link Category#COMMUNITY} while it depends *during runtime* (scopes are defined within
   * {@link ResolutionScope#COMPILE}) in EE components or EE mule-plugins.
   */
  private Product calculateProduct(String name, String category) throws MojoExecutionException {
    Product product = isBlank(category) || COMMUNITY.name().equals(category) ? MULE : MULE_EE;
    //only looking for EE artifacts that are not mule-plugins
    List<Artifact> eeArtifacts =
        project.getArtifacts().stream()
            .filter(artifact -> artifact.getGroupId().startsWith(EE_MULESOFT_ARTIFACTS_GROUP_ID_PREFIX)
                && !MULE_PLUGIN_CLASSIFIER.equals(artifact.getClassifier()))
            .collect(Collectors.toList());
    //only looking for mule-plugins which happens to require EE runtimes
    List<Artifact> mulePluginEEArtifacts =
        project.getArtifacts().stream()
            .filter(artifact -> readMulePluginModel(artifact)
                .map(mulePluginModel -> mulePluginModel.getRequiredProduct().supports(MULE_EE)).orElse(false))
            .collect(Collectors.toList());

    if ((!eeArtifacts.isEmpty() || !mulePluginEEArtifacts.isEmpty())) {
      if (!product.equals(MULE)) {
        product = MULE_EE;
      } else {
        //the module states it's CE while transitively it is not
        StringBuilder sb = new StringBuilder("The current module states it's ").append(COMMUNITY)
            .append(" while it relies in EE components/EE Mule Plugins at runtime directly or indirectly (through transitive dependencies).");
        if (!eeArtifacts.isEmpty()) {
          sb.append("\nFull list of EE components:\n")
              .append(String.join(",\n", eeArtifacts.stream().map(Object::toString).collect(Collectors.toSet())));
        }
        if (!mulePluginEEArtifacts.isEmpty()) {
          sb.append("\nFull list of EE Mule Plugins:\n")
              .append(String.join(",\n", mulePluginEEArtifacts.stream().map(Object::toString).collect(Collectors.toSet())));
        }
        String templateModuleElement = "<module name=\"" + name + "\" category=\"%s\" .../>";
        sb.append("\nUpdate, or add, the 'category' attribute of the module element to any of the following:\n")
            .append(format(templateModuleElement, SELECT)).append(",\n")
            .append(format(templateModuleElement, PREMIUM)).append(",\n")
            .append(format(templateModuleElement, CERTIFIED)).append(".")
            .append("\nFor more details please go to https://docs.mulesoft.com/mule-sdk/v/1.1/license");
        throw new MojoExecutionException(sb.toString());
      }
    }
    return product;
  }

  /**
   * @param extensionName extension name
   * @param moduleDocument entire document to extract values
   * @return collection of relative files names that needs to be exported
   * @throws MojoExecutionException if there issues while reading files
   */
  private Set<String> calculateExportedResources(String extensionName, Document moduleDocument) throws MojoExecutionException {
    final Set<String> result = new HashSet<>();

    final Optional<String> prefix = ofNullable(moduleDocument.getDocumentElement().getAttribute(PREFIX_ATTRIBUTE));
    final XmlDslModel xmlLanguageModel = createXmlLanguageModel(prefix, Optional.empty(), extensionName, null);
    final String sanitizedName = sanitizeName(xmlLanguageModel.getPrefix());

    final Path targetResourceFolder = Paths.get(project.getBuild().getOutputDirectory(), sanitizedName);
    final Set<String> processedResources = findFilesFrom(targetResourceFolder, sanitizedName);
    for (Resource resource : project.getResources()) {
      Set<String> resourcesToExport = calculateExportedResources(resource, processedResources, sanitizedName)
          .stream()
          .map(MulePluginArtifactLoaderUtils::toZipPath)
          .collect(Collectors.toSet());
      result.addAll(resourcesToExport);
    }
    return result;
  }

  /**
   * @param name an extension prefix such as 'sfdc', 'github', or 'module-using-core' that will be sanitized for further references
   * @return a sanitized name, 'sfdc' -> 'sfdc', 'github' -> 'github', or 'module-using-core' -> 'module_using_core'
   */
  private String sanitizeName(String name) {
    char[] nameArray = name.toCharArray();
    for (int i = 0; i < nameArray.length; i++) {
      if (!Character.isJavaIdentifierPart(nameArray[i])) {
        nameArray[i] = '_';
      }
    }
    return ((!Character.isJavaIdentifierStart(nameArray[0])) ? "_" : "") + new String(nameArray);
  }

  /**
   * @param resource a starting point to look for files that might need to be exported
   * @param processedResources already copied resources by the maven-resources-plugin
   * @param prefixFolder delimiter of elements that will be exported out of the box
   * @return a collection of relative file elements that must be exported as resources, as they belong to the resources of the
   * project as well as in the output directory (aka: the /target), implying that the maven-resources-plugin has copied them
   * applying the correct filters (inclusions, exclusions, etc.)
   *
   * @throws MojoExecutionException if there are any issues reading the directories.
   */
  private Set<String> calculateExportedResources(Resource resource, Set<String> processedResources,
                                                 String prefixFolder)
      throws MojoExecutionException {
    Set<String> result = new HashSet<>();
    Path resouceWithExportFolder = Paths.get(resource.getDirectory(), prefixFolder);
    Set<String> resourceFiles = findFilesFrom(resouceWithExportFolder, prefixFolder);
    if (resourceFiles.isEmpty()) {
      getLog().info(format("No resources to export in [%s] (only resources under [%s] folder will be automatically exported)",
                           resource.getDirectory(), prefixFolder));
    } else {
      getLog().debug(format("Looking for resources under [%s] to export", resouceWithExportFolder));
      final Set<String> notExportables = new HashSet<>();
      final Set<String> exportables = new HashSet<>();
      resourceFiles
          .stream()
          .filter(processedResources::contains)
          .forEach(path -> {
            if (!path.toLowerCase().contains(INTERNAL_FOLDER)) {
              exportables.add(path);

            } else {
              notExportables.add(path);
            }
          });

      if (!notExportables.isEmpty()) {
        getLog()
            .info(format(
                         "The following files under [%s] won't be exported as they are inside folders that contain the [%s] value. Those files are: [%s]",
                         resouceWithExportFolder, INTERNAL_FOLDER, String.join(", ", notExportables)));
      }
      getLog()
          .info(format("Found [%s] resource%s to export under [%s]", exportables.size(), exportables.size() > 1 ? "s" : "",
                       resouceWithExportFolder));
      result.addAll(exportables);
    }
    return result;
  }

  /**
   * @param originPath starting point to look for files in the resources folder
   * @param prefixFolder adding the parent folder to each element to be sure it's correctly saved into the mule-artifact.json
   *                     exportedResources file
   * @return a collection of relative filenames from the original {@code originPath}
   * @throws MojoExecutionException if there's an issue while reading the directory under {@code originPath}
   */
  private Set<String> findFilesFrom(Path originPath, String prefixFolder) throws MojoExecutionException {
    Set<String> result = new HashSet<>();

    if (Files.exists(originPath)) {
      try {
        Files.walk(originPath)
            .filter(path -> isRegularFile(path))
            .map(file -> prefixFolder + separator + originPath.relativize(file).toString())
            .forEach(result::add);
      } catch (IOException e) {
        throw new MojoExecutionException(format("There was an issue looking for resources in the folder [%s]",
                                                originPath),
                                         e);
      }
    }
    return result;
  }

  /**
   * Looks in {@code baseDirectory} recursively looking for files that start with {@link #PREFIX_SMART_CONNECTOR_NAME} and ends
   * with {@link #SUFFIX_SMART_CONNECTOR_NAME} to then validate through {@link #getModule(File)} if they are a <module/> or not.
   *
   * @param baseDirectory base path to start looking for <module/>
   * @return a {@link File} that targets the only <module/> of the current project.
   * @throws MojoExecutionException if the amount of <module/>s found is different than 1. There can be only one, Highlander.
   */
  private File getModuleFile(String baseDirectory) throws MojoExecutionException {
    final List<File> modulesFiles = listFiles(new File(baseDirectory), new FileFileFilter() {

      @Override
      public boolean accept(File file) {
        return file.getName().startsWith(PREFIX_SMART_CONNECTOR_NAME)
            && file.getName().endsWith(SUFFIX_SMART_CONNECTOR_NAME);
      }
    }, TrueFileFilter.INSTANCE)
        .stream()
        .filter(file -> getModule(file).isPresent())
        .collect(Collectors.toList());

    if (modulesFiles.size() > 1) {
      final String xmlModules = String.join("\n,", modulesFiles.stream().map(File::getAbsolutePath).collect(Collectors.toList()));
      throw new MojoExecutionException(format(
                                              "There are several XML files that have a <module> root element, when there must be only one. Files with <module> as root element are: [%s]",
                                              xmlModules));
    } else if (modulesFiles.isEmpty()) {
      throw new MojoExecutionException(format(
                                              "There's no XML files that has a <module> root element, thus is impossible to auto generate a [%s] descriptor file. The file must start with [%s] and end with [%s], such as [%s]",
                                              MULE_ARTIFACT_JSON,
                                              PREFIX_SMART_CONNECTOR_NAME,
                                              SUFFIX_SMART_CONNECTOR_NAME,
                                              PREFIX_SMART_CONNECTOR_NAME + "foo" + SUFFIX_SMART_CONNECTOR_NAME));
    }
    return modulesFiles.get(0);
  }

  /**
   * Given an existing {@link File}, it returns a {@link Document} if it's able to parse it and the root element's name matches
   * with {@link #MODULE_ROOT_ELEMENT}.
   *
   * @param file to be read
   * @return the parsed file if it's a <module/>
   */
  private Optional<Document> getModule(File file) {
    Optional<Document> result = empty();
    DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
    try {
      DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();
      Document doc = dBuilder.parse(file);
      if (doc.getDocumentElement().getNodeName().equals(MODULE_ROOT_ELEMENT)) {
        result = of(doc);
      }
    } catch (ParserConfigurationException | SAXException | IOException e) {
      // If it fails, then the file wasn't a <module> after all :)
    }
    return result;
  }
}
