/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.extension.maven.loader;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static java.lang.String.format;
import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.runtime.core.api.registry.SpiServiceRegistry;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptorLoader;
import org.mule.runtime.module.artifact.api.descriptor.ClassLoaderModelLoader;
import org.mule.runtime.module.artifact.api.descriptor.DescriptorLoader;
import org.mule.runtime.module.artifact.api.descriptor.DescriptorLoaderRepository;
import org.mule.runtime.module.artifact.api.descriptor.LoaderNotFoundException;
import org.mule.runtime.module.deployment.impl.internal.application.DeployableMavenClassLoaderModelLoader;
import org.mule.runtime.module.deployment.impl.internal.plugin.PluginMavenClassLoaderModelLoader;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

public class MavenDescriptorLoaderRepository implements DescriptorLoaderRepository {

  private Map<Class, List<DescriptorLoader>> descriptorLoaders = newHashMap();

  public MavenDescriptorLoaderRepository(MavenClient mavenClient) {
    CompositeClassLoaderModelLoader compositeClassLoaderModelLoader =
        new CompositeClassLoaderModelLoader(newArrayList(new DeployableMavenClassLoaderModelLoader(mavenClient),
                                                         new PluginMavenClassLoaderModelLoader(mavenClient)));
    descriptorLoaders.put(BundleDescriptorLoader.class,
                          findBundleDescriptorLoaders(BundleDescriptorLoader.class, new SpiServiceRegistry()));
    descriptorLoaders.put(ClassLoaderModelLoader.class, newArrayList(compositeClassLoaderModelLoader));
  }

  private List<DescriptorLoader> findBundleDescriptorLoaders(Class<? extends DescriptorLoader> descriptorLoaderClass,
                                                             SpiServiceRegistry serviceRegistry) {
    List<DescriptorLoader> found = new ArrayList<>();
    Collection<? extends DescriptorLoader> providers =
        serviceRegistry.lookupProviders(descriptorLoaderClass, this.getClass().getClassLoader());

    for (DescriptorLoader loader : providers) {
      found.add(loader);
    }

    return found;
  }

  @Override
  public <T extends DescriptorLoader> T get(String id, ArtifactType artifactType, Class<T> loaderClass)
      throws LoaderNotFoundException {
    DescriptorLoader descriptorLoader = null;
    List<DescriptorLoader> registeredDescriptorLoaders = descriptorLoaders.get(loaderClass);
    if (registeredDescriptorLoaders != null) {
      for (DescriptorLoader loader : registeredDescriptorLoaders) {
        if (loader.getId().equals(id) && loader.supportsArtifactType(artifactType)) {
          descriptorLoader = loader;
        }
      }
    }

    if (descriptorLoader == null) {
      throw new LoaderNotFoundException(format("There is no loader with ID='%s' and type '%s'", id, loaderClass.getName()));
    }

    return (T) descriptorLoader;
  }
}
