/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extension.maven;

import static java.lang.String.format;

import org.mule.extension.maven.loader.MavenProjectExtensionModelLoader;
import org.mule.plugin.maven.AbstractMuleMojo;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.extension.api.persistence.ExtensionModelJsonSerializer;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.util.List;

import org.apache.maven.artifact.DefaultArtifact;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.resolution.ArtifactRequest;
import org.eclipse.aether.resolution.ArtifactResolutionException;

/**
 * This goal will resolve the requested Mule Plugin using project's remote repositories to download the artifact and load its
 * {@link ExtensionModel}, once loaded it will be written to a file.
 * <p/>
 * This {@link Mojo} must be executed on Maven project as it uses the remote repositories resolved by the project and it is not
 * possible to provide the list of remote repositories by user's configuration parameter.
 *
 * @since 1.3.2
 */
@Mojo(name = "extension-model-load", threadSafe = true)
public class ExtensionModelLoadMojo extends AbstractMuleMojo {

  private static final String MULE_PLUGIN = "mule-plugin";

  @Parameter(defaultValue = "${session}", readonly = true, required = true)
  private MavenSession session;

  @Component
  protected RepositorySystem repositorySystem;

  @Parameter(defaultValue = "${repositorySystemSession}", readonly = true)
  protected RepositorySystemSession repositorySystemSession;

  /**
   * The project's remote repositories to use for the resolution of Mule Plugin artifact.
   */
  @Parameter(defaultValue = "${project.remoteProjectRepositories}", readonly = true)
  private List<RemoteRepository> remoteRepositories;

  /**
   * The output file name for the generated extension model json to be written. Default value is
   * {@code ${artifactId}-${version}-extension-model.json}. Note: It cannot be defined with references to other parameters from
   * this MOJO when a value is set in pom.xml for this parameter.
   */
  @Parameter
  private String outputFileName;

  /**
   * The output directory for the generated extension model json to be written.
   */
  @Parameter(defaultValue = "${project.build.directory}")
  private File outputDirectory;

  /**
   * {@groupId} for the Mule Plugin artifact to resolve its extension model.
   */
  @Parameter(required = true)
  private String groupId;

  /**
   * {@code artifactId} for the Mule Plugin artifact to resolve its extension model.
   */
  @Parameter(required = true)
  private String artifactId;

  /**
   * {@code version} for the Mule Plugin artifact to resolve its extension model.
   */
  @Parameter(required = true)
  private String version;

  @Override
  public void execute() throws MojoExecutionException, MojoFailureException {
    getLog().info(format("Loading ExtensionModel for artifact [%s:%s:%s]", groupId, artifactId, version));

    org.apache.maven.artifact.Artifact artifact = resolveMulePluginArtifact();

    MavenProjectExtensionModelLoader extensionModelLoader = new MavenProjectExtensionModelLoader(getLog());
    final ExtensionModel extensionModel = extensionModelLoader.loadExtension(artifact, session);

    final String serializedExtensionModel = new ExtensionModelJsonSerializer(true).serialize(extensionModel);

    outputDirectory.mkdirs();
    File outputFile = new File(outputDirectory, outputFileName(artifact));

    try {
      try (PrintWriter out = new PrintWriter(outputFile)) {
        out.println(serializedExtensionModel);
      }
    } catch (FileNotFoundException e) {
      throw new MojoFailureException(
                                     format("Failure while saving the serialized ExtensionModel to the file [%s]",
                                            outputFile.getAbsolutePath()),
                                     e);
    }
  }

  private String outputFileName(org.apache.maven.artifact.Artifact artifact) {
    if (outputFileName != null) {
      return outputFileName;
    }
    return artifact.getArtifactId() + "-" +
        artifact.getVersion() + "-" +
        "extension-model.json";
  }

  private org.apache.maven.artifact.Artifact resolveMulePluginArtifact() throws MojoExecutionException {
    org.eclipse.aether.artifact.DefaultArtifact aetherArtifact =
        new org.eclipse.aether.artifact.DefaultArtifact(groupId, artifactId,
                                                        MULE_PLUGIN, "jar", version);
    try {
      ArtifactRequest artifactRequest =
          new ArtifactRequest(aetherArtifact,
                              remoteRepositories, null);
      Artifact artifact =
          repositorySystem.resolveArtifact(repositorySystemSession, artifactRequest).getArtifact();
      return new DefaultArtifact(artifact.getGroupId(), artifact.getArtifactId(), artifact.getVersion(), null,
                                 artifact.getExtension(), artifact.getClassifier(), null);
    } catch (ArtifactResolutionException e) {
      throw new MojoExecutionException(format("Error while resolving Mule Plugin artifact: [%s]",
                                              aetherArtifact),
                                       e);
    }
  }

}
