/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.extension.maven.loader;

import static com.google.common.io.Files.createTempDir;
import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.mule.extension.maven.ExtensionDescriptorMojo.XML_BASED_EXTENSION_MODEL_LOADER;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.core.internal.util.JarUtils.loadFileContentFrom;
import static org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor.MULE_ARTIFACT_PATH_INSIDE_JAR;
import static org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptor.MULE_ARTIFACT_JSON_DESCRIPTOR;
import org.mule.extension.maven.ExtensionDescriptorMojo;
import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptor;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptorBuilder;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.deployment.meta.Product;
import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.container.internal.ClasspathModuleDiscoverer;
import org.mule.runtime.container.internal.CompositeModuleDiscoverer;
import org.mule.runtime.container.internal.ContainerClassLoaderFactory;
import org.mule.runtime.container.internal.DefaultModuleRepository;
import org.mule.runtime.container.internal.JreModuleDiscoverer;
import org.mule.runtime.container.internal.ModuleDiscoverer;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginClassLoaderFactory;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.deployment.model.internal.DefaultRegionPluginClassLoadersFactory;
import org.mule.runtime.deployment.model.internal.RegionPluginClassLoadersFactory;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptorCreateException;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptorValidatorBuilder;
import org.mule.runtime.module.artifact.api.descriptor.DescriptorLoaderRepository;
import org.mule.runtime.module.deployment.impl.internal.application.ToolingApplicationDescriptorFactory;
import org.mule.runtime.module.deployment.impl.internal.plugin.ArtifactPluginDescriptorFactory;
import org.mule.runtime.module.deployment.impl.internal.plugin.BundlePluginDependenciesResolver;
import org.mule.runtime.module.deployment.impl.internal.plugin.PluginExtendedDeploymentProperties;

import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Properties;
import java.util.function.Predicate;

import org.apache.commons.io.FileUtils;
import org.apache.maven.project.MavenProject;

public class MuleArtifactResourcesRegistry {

  private ToolingApplicationDescriptorFactory applicationDescriptorFactory;
  private RegionPluginClassLoadersFactory regionPluginClassLoadersFactory;
  private BundlePluginDependenciesResolver pluginDependenciesResolver;
  private ArtifactClassLoader containerArtifactClassLoader;


  public MuleArtifactResourcesRegistry(MavenClient mavenClient, Predicate<PluginDescriptor> enableXmlValidations) {

    DescriptorLoaderRepository descriptorLoaderRepository = new MavenDescriptorLoaderRepository(mavenClient);
    ModuleRepository moduleRepository = createModuleRepository();

    ArtifactDescriptorValidatorBuilder artifactDescriptorValidatorBuilder = ArtifactDescriptorValidatorBuilder.builder();
    ArtifactPluginDescriptorFactory artifactPluginDescriptorFactory =
        new ArtifactPluginDescriptorFactory(descriptorLoaderRepository, artifactDescriptorValidatorBuilder) {

          /**
           * Custom implementation that won't fail if the pluginFile is not a jar/zip file. Works with target/classes too.
           */
          @Override
          public ArtifactPluginDescriptor create(File pluginFile, Optional<Properties> deploymentProperties) {
            try {
              Optional<byte[]> jsonDescriptorContentOptional = empty();
              String mulePluginJsonPath = MULE_ARTIFACT_PATH_INSIDE_JAR + "/" + MULE_ARTIFACT_JSON_DESCRIPTOR;
              jsonDescriptorContentOptional =
                  getMuleArtifactJsonDescriptorContent(pluginFile, jsonDescriptorContentOptional, mulePluginJsonPath);

              MulePluginModel artifactModel = jsonDescriptorContentOptional
                  .map(jsonDescriptorContent -> loadModelFromJson(new String(jsonDescriptorContent)))
                  .orElseThrow(() -> new ArtifactDescriptorCreateException(format("The plugin descriptor '%s' on plugin file '%s' is not present",
                                                                                  mulePluginJsonPath, pluginFile)));

              // Create another plugin model to set as required product MULE and minMuleVersion 4.1.1 due to validations cannot be
              // disabled
              MulePluginModel.MulePluginModelBuilder mulePluginModelBuilder = new MulePluginModel.MulePluginModelBuilder();
              mulePluginModelBuilder
                  .setName(artifactModel.getName())
                  .setMinMuleVersion("4.1.1")
                  .setRequiredProduct(Product.MULE)
                  .withBundleDescriptorLoader(artifactModel.getBundleDescriptorLoader())
                  .withClassLoaderModelDescriptorLoader(artifactModel.getClassLoaderModelLoaderDescriptor());

              artifactModel.getLicense().ifPresent(licenseModel -> mulePluginModelBuilder.withLicenseModel()
                  .setAllowsEvaluationLicense(licenseModel.isAllowsEvaluation())
                  .setProvider(licenseModel.getProvider())
                  .setRequiredEntitlement(licenseModel.getRequiredEntitlement().orElse(null)));

              artifactModel.getExtensionModelLoaderDescriptor().ifPresent(extensionModelDescriptorLoader -> {
                MuleArtifactLoaderDescriptorBuilder muleArtifactLoaderDescriptorBuilder =
                    mulePluginModelBuilder.withExtensionModelDescriber();
                muleArtifactLoaderDescriptorBuilder.setId(extensionModelDescriptorLoader.getId());
                extensionModelDescriptorLoader.getAttributes().entrySet().stream()
                    .forEach(entry -> muleArtifactLoaderDescriptorBuilder.addProperty(entry.getKey(), entry.getValue()));

                // If the current plugin being built is an XML SDK we should always validate its XML
                if (artifactModel.getExtensionModelLoaderDescriptor().isPresent() &&
                    enableXmlValidations
                        .test(new PluginDescriptor(pluginFile, artifactModel.getExtensionModelLoaderDescriptor().get()))) {
                  muleArtifactLoaderDescriptorBuilder.addProperty(ExtensionDescriptorMojo.VALIDATE_XML, true);
                }
              });

              MulePluginModel pluginModel = mulePluginModelBuilder.build();
              return this.loadFromJsonDescriptor(pluginFile, pluginModel, deploymentProperties);
            } catch (ArtifactDescriptorCreateException e) {
              throw e;
            } catch (IOException e) {
              throw new ArtifactDescriptorCreateException(e);
            }
          }

          private Optional<byte[]> getMuleArtifactJsonDescriptorContent(File pluginFile,
                                                                        Optional<byte[]> jsonDescriptorContentOptional,
                                                                        String mulePluginJsonPath)
              throws IOException {
            if (pluginFile.isFile()) {
              checkArgument(pluginFile.getName().endsWith(".jar"),
                            "provided file is not a plugin: " + pluginFile.getAbsolutePath());
              jsonDescriptorContentOptional = loadFileContentFrom(pluginFile, mulePluginJsonPath);
            } else {
              File mulePluginJsonFile =
                  pluginFile.toPath().resolve(Paths.get(MULE_ARTIFACT_PATH_INSIDE_JAR, MULE_ARTIFACT_JSON_DESCRIPTOR)).toFile();
              if (mulePluginJsonFile.exists()) {
                jsonDescriptorContentOptional = of(FileUtils.readFileToByteArray(mulePluginJsonFile));
              }
            }
            return jsonDescriptorContentOptional;
          }
        };
    org.mule.runtime.module.deployment.impl.internal.plugin.ArtifactPluginDescriptorLoader artifactPluginDescriptorLoader =
        new ArtifactPluginDescriptorLoaderMojo(artifactPluginDescriptorFactory);

    applicationDescriptorFactory =
        new ToolingApplicationDescriptorFactory(artifactPluginDescriptorLoader, descriptorLoaderRepository,
                                                artifactDescriptorValidatorBuilder);


    regionPluginClassLoadersFactory =
        new DefaultRegionPluginClassLoadersFactory(new ArtifactPluginClassLoaderFactory(), moduleRepository);

    pluginDependenciesResolver = new BundlePluginDependenciesResolver(artifactPluginDescriptorFactory);

    containerArtifactClassLoader = createContainerClassLoader(this.getClass().getClassLoader(), moduleRepository);

  }

  private ModuleRepository createModuleRepository() {
    List<ModuleDiscoverer> result = new ArrayList<>();
    result.add(new JreModuleDiscoverer());
    result.add(new ClasspathModuleDiscoverer(this.getClass().getClassLoader(), createTempDir()));
    return new DefaultModuleRepository(new CompositeModuleDiscoverer(result.toArray(new ModuleDiscoverer[0])));
  }

  /**
   * Implementation that supports to resolve a plugin descriptor from compiled classes and not only from a packaged artifact as
   * this MOJO runs before artifact is packaged.
   */
  class ArtifactPluginDescriptorLoaderMojo
      extends org.mule.runtime.module.deployment.impl.internal.plugin.ArtifactPluginDescriptorLoader {

    private final ArtifactPluginDescriptorFactory artifactPluginDescriptorFactory;

    public ArtifactPluginDescriptorLoaderMojo(ArtifactPluginDescriptorFactory artifactPluginDescriptorFactory) {
      super(artifactPluginDescriptorFactory);
      this.artifactPluginDescriptorFactory = artifactPluginDescriptorFactory;
    }

    @Override
    public ArtifactPluginDescriptor load(File plugin) {
      return this.artifactPluginDescriptorFactory.create(plugin, empty());
    }

    @Override
    public ArtifactPluginDescriptor load(File pluginFile,
                                         org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor pluginBundleDescriptor,
                                         ArtifactDescriptor deployableArtifactDescriptor) {
      return this.artifactPluginDescriptorFactory
          .create(pluginFile, of(new PluginExtendedDeploymentProperties(new Properties(), pluginBundleDescriptor,
                                                                        deployableArtifactDescriptor)));
    }
  }

  public ArtifactClassLoader createContainerClassLoader(ClassLoader pluginClassLoader, ModuleRepository moduleRepository) {
    ArtifactClassLoader containerClassLoaderFactory;
    containerClassLoaderFactory =
        new ContainerClassLoaderFactory(moduleRepository)
            .createContainerClassLoader(pluginClassLoader);
    return containerClassLoaderFactory;
  }

  public ToolingApplicationDescriptorFactory getApplicationDescriptorFactory() {
    return applicationDescriptorFactory;
  }

  public RegionPluginClassLoadersFactory getRegionPluginClassLoadersFactory() {
    return regionPluginClassLoadersFactory;
  }

  public BundlePluginDependenciesResolver getPluginDependenciesResolver() {
    return pluginDependenciesResolver;
  }

  public ArtifactClassLoader getContainerArtifactClassLoader() {
    return containerArtifactClassLoader;
  }

  public static class PluginDescriptor {

    private File pluginFile;
    private MuleArtifactLoaderDescriptor muleArtifactLoaderDescriptor;

    public PluginDescriptor(File pluginFile, MuleArtifactLoaderDescriptor muleArtifactLoaderDescriptor) {
      this.pluginFile = pluginFile;
      this.muleArtifactLoaderDescriptor = muleArtifactLoaderDescriptor;
    }

    public File getPluginFile() {
      return pluginFile;
    }

    public MuleArtifactLoaderDescriptor getMuleArtifactLoaderDescriptor() {
      return muleArtifactLoaderDescriptor;
    }
  }
}
