/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.extension.maven.loader;

import static java.lang.String.format;
import static java.lang.String.join;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.maven.artifact.ArtifactUtils.key;
import static org.mule.extension.maven.ExtensionDescriptorMojo.XML_BASED_EXTENSION_MODEL_LOADER;
import static org.mule.plugin.maven.AbstractPackagePluginMojo.MULE_PLUGIN_CLASSIFIER;
import static org.mule.runtime.api.deployment.meta.Product.MULE;
import static org.mule.runtime.core.api.config.MuleManifest.getProductVersion;
import static org.mule.runtime.core.api.util.UUID.getUUID;
import static org.mule.runtime.deployment.model.api.artifact.ArtifactDescriptorConstants.MULE_LOADER_ID;
import static org.mule.runtime.globalconfig.api.GlobalConfigLoader.setMavenConfig;
import static org.mule.runtime.module.deployment.impl.internal.maven.AbstractMavenClassLoaderModelLoader.CLASSLOADER_MODEL_MAVEN_REACTOR_RESOLVER;
import static org.mule.runtime.module.deployment.impl.internal.maven.MavenUtils.createDeployablePomFile;
import static org.mule.runtime.module.deployment.impl.internal.maven.MavenUtils.createDeployablePomProperties;

import org.mule.maven.client.api.MavenReactorResolver;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.DefaultSettingsSupplierFactory;
import org.mule.maven.client.internal.MavenEnvironmentVariables;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptor;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.deployment.model.api.artifact.extension.ExtensionModelDiscoverer;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.deployment.model.internal.artifact.extension.MuleExtensionModelLoaderManager;
import org.mule.runtime.deployment.model.internal.tooling.ToolingApplicationClassLoaderBuilder;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.runtime.extension.api.loader.ExtensionModelLoader;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.DeployableArtifactClassLoaderFactory;
import org.mule.runtime.module.artifact.api.classloader.MuleDeployableArtifactClassLoader;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.google.common.io.Files;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.execution.MavenExecutionRequest;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.Model;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

/**
 * Loader of an {@link ExtensionModel} for a Mule plugin artifact from an extension {@link MavenProject}.
 *
 * @since 1.0
 */
public class MavenProjectExtensionModelLoader {

  private final Log log;

  public MavenProjectExtensionModelLoader(Log log) {
    this.log = log;
  }

  /**
   * Builds an {@link ExtensionModel} from a {@link MavenProject} and a {@link MulePluginModel} that holds the information about
   * the extension mule plugin that is wanted to be loaded.
   *
   * @return an {@link ExtensionModel} built up from the extension {@link MavenProject} provided.
   * @throws MojoFailureException
   */
  public ExtensionModel loadExtension(MavenProject project, MavenSession session)
      throws MojoFailureException, MojoExecutionException {
    String artifactId = project.getArtifactId();
    log.debug(format("Creating ExtensionModel for:[%s]", artifactId));
    configureMavenClient(session);

    MuleArtifactResourcesRegistry muleArtifactResourcesRegistry =
        new MuleArtifactResourcesRegistry(pluginDescriptor -> pluginDescriptor.getPluginFile()
            .equals(project.getArtifact().getFile()) &&
            XML_BASED_EXTENSION_MODEL_LOADER.equals(pluginDescriptor.getMuleArtifactLoaderDescriptor().getId()));

    Map<String, Object> classLoaderModelAttributes = new HashMap<>();
    classLoaderModelAttributes.put(CLASSLOADER_MODEL_MAVEN_REACTOR_RESOLVER, new PluginFileMavenReactor(project));

    // TODO review this project.getArtifact() vs project.getGroupId()... etc (which uses the parent pom if not a value at the pom
    // model level)
    return withPluginClassLoaders(project.getArtifact(), classLoaderModelAttributes, muleArtifactResourcesRegistry,
                                  artifactPluginClassLoaders -> doLoadExtensionModel(project.getArtifact(),
                                                                                     artifactPluginClassLoaders,
                                                                                     muleArtifactResourcesRegistry
                                                                                         .getContainerArtifactClassLoader()));
  }

  public ExtensionModel loadExtension(Artifact artifact, MavenSession session)
      throws MojoFailureException, MojoExecutionException {
    log.debug(format("Creating ExtensionModel for:[%s]", artifact.getArtifactId()));
    configureMavenClient(session);

    MuleArtifactResourcesRegistry muleArtifactResourcesRegistry = new MuleArtifactResourcesRegistry(pluginDescriptor -> false);

    Map<String, Object> classLoaderModelAttributes = new HashMap<>();

    return withPluginClassLoaders(artifact, classLoaderModelAttributes, muleArtifactResourcesRegistry,
                                  artifactPluginClassLoaders -> doLoadExtensionModel(artifact,
                                                                                     artifactPluginClassLoaders,
                                                                                     muleArtifactResourcesRegistry
                                                                                         .getContainerArtifactClassLoader()));
  }

  private Set<org.mule.runtime.api.util.Pair<ArtifactPluginDescriptor, ExtensionModel>> discoverPluginsExtensionModel(List<ArtifactClassLoader> artifactPluginClassLoaders,
                                                                                                                      MuleExtensionModelLoaderManager extensionModelLoaderRepository) {
    List<org.mule.runtime.api.util.Pair<ArtifactPluginDescriptor, ArtifactClassLoader>> artifacts = artifactPluginClassLoaders
        .stream()
        .map(a -> new org.mule.runtime.api.util.Pair<ArtifactPluginDescriptor, ArtifactClassLoader>(a.getArtifactDescriptor(), a))
        .collect(toList());
    return new ExtensionModelDiscoverer().discoverPluginsExtensionModels(extensionModelLoaderRepository, artifacts);
  }


  private void configureMavenClient(MavenSession session) {
    MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder = MavenConfiguration.newMavenConfigurationBuilder();
    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(new MavenEnvironmentVariables());
    MavenExecutionRequest request = session.getRequest();

    mavenConfigurationBuilder.localMavenRepositoryLocation(request.getLocalRepositoryPath());

    mavenConfigurationBuilder.userSettingsLocation(request.getUserSettingsFile());
    mavenConfigurationBuilder.globalSettingsLocation(request.getGlobalSettingsFile());
    settingsSupplierFactory.environmentSettingsSecuritySupplier().ifPresent(mavenConfigurationBuilder::settingsSecurityLocation);

    mavenConfigurationBuilder.userProperties(request.getUserProperties());
    mavenConfigurationBuilder.activeProfiles(request.getActiveProfiles());
    mavenConfigurationBuilder.inactiveProfiles(request.getInactiveProfiles());

    mavenConfigurationBuilder.ignoreArtifactDescriptorRepositories(false);

    setMavenConfig(mavenConfigurationBuilder.build());
  }

  class PluginFileMavenReactor implements MavenReactorResolver {

    private final MavenProject project;

    public PluginFileMavenReactor(MavenProject project) {
      this.project = project;
    }

    @Override
    public File findArtifact(BundleDescriptor bundleDescriptor) {
      if (checkArtifact(bundleDescriptor)) {
        if (bundleDescriptor.getType().equals("pom")) {
          return project.getFile();
        } else {
          return project.getArtifact().getFile();
        }
      }
      return null;
    }

    private boolean checkArtifact(BundleDescriptor bundleDescriptor) {
      return bundleDescriptor.getGroupId().equals(project.getArtifact().getGroupId())
          && bundleDescriptor.getArtifactId().equals(project.getArtifact().getArtifactId())
          && bundleDescriptor.getVersion().equals(project.getArtifact().getVersion());
    }

    @Override
    public List<String> findVersions(BundleDescriptor bundleDescriptor) {
      if (checkArtifact(bundleDescriptor)) {
        return singletonList(project.getArtifact().getVersion());
      }
      return emptyList();
    }
  }

  private ExtensionModel doLoadExtensionModel(Artifact artifact, List<ArtifactClassLoader> artifactPluginClassLoaders,
                                              ArtifactClassLoader containerArtifactClassLoader)
      throws MojoFailureException {
    MuleExtensionModelLoaderManager extensionModelLoaderRepository =
        new MuleExtensionModelLoaderManager(containerArtifactClassLoader);
    try {
      extensionModelLoaderRepository.start();
    } catch (MuleException e) {
      throw new MojoFailureException(format("Failure while looking implementation classes of [%s] class through SPI",
                                            ExtensionModelLoader.class.getName()),
                                     e);
    }
    Set<Pair<ArtifactPluginDescriptor, ExtensionModel>> discoverPluginsExtensionModel =
        discoverPluginsExtensionModel(artifactPluginClassLoaders, extensionModelLoaderRepository);
    Optional<ExtensionModel> extensionModelOptional = discoverPluginsExtensionModel.stream()
        .filter(pair -> {
          org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor bundleDescriptor =
              pair.getFirst().getBundleDescriptor();
          return bundleDescriptor.getGroupId().equals(artifact.getGroupId()) &&
              bundleDescriptor.getArtifactId().equals(artifact.getArtifactId()) &&
              bundleDescriptor.getVersion().equals(artifact.getVersion());
        })
        .findFirst()
        .map(Pair::getSecond);

    if (!extensionModelOptional.isPresent()) {
      List<String> foundExtensionModels =
          discoverPluginsExtensionModel.stream().map(pair -> pair.getSecond().getName()).collect(toList());
      if (foundExtensionModels.isEmpty()) {
        throw new MojoFailureException(format("Could not load Extension Model for %s", key(artifact)));
      }
      throw new MojoFailureException(format("Could not load Extension Model for %s, the following were loaded: %n%s",
                                            key(artifact), join(format(", %n"), foundExtensionModels)));
    }

    return extensionModelOptional.get();
  }

  private <T> T withPluginClassLoaders(Artifact artifact, Map<String, Object> classLoaderModelLoaderAttributes,
                                       MuleArtifactResourcesRegistry muleArtifactResourcesRegistry,
                                       Action<T> action)
      throws MojoFailureException, MojoExecutionException {
    String uuid = getUUID();
    String applicationName = uuid + "-extension-model-temp-app";
    File applicationFolder = new File(Files.createTempDir(), applicationName);
    try {
      createPomFile(artifact, uuid, applicationFolder);

      MuleApplicationModel muleApplicationModel = new MuleApplicationModel.MuleApplicationModelBuilder()
          .setMinMuleVersion(getProductVersion())
          .setName(applicationName)
          .setRequiredProduct(MULE)
          .withBundleDescriptorLoader(new MuleArtifactLoaderDescriptor(MULE_LOADER_ID, emptyMap()))
          .withClassLoaderModelDescriptorLoader(new MuleArtifactLoaderDescriptor(MULE_LOADER_ID,
                                                                                 classLoaderModelLoaderAttributes))
          .build();
      ApplicationDescriptor artifactDescriptor = muleArtifactResourcesRegistry.getApplicationDescriptorFactory()
          .createArtifact(applicationFolder, empty(), muleApplicationModel);

      ToolingApplicationClassLoaderBuilder builder =
          new ToolingApplicationClassLoaderBuilder(newTemporaryArtifactClassLoaderFactory(),
                                                   muleArtifactResourcesRegistry.getRegionPluginClassLoadersFactory());
      builder.setArtifactDescriptor(artifactDescriptor);
      builder.setParentClassLoader(muleArtifactResourcesRegistry.getContainerArtifactClassLoader());

      muleArtifactResourcesRegistry.getPluginDependenciesResolver()
          .resolve(emptySet(), new ArrayList<>(artifactDescriptor.getPlugins()), false)
          .stream()
          .forEach(builder::addArtifactPluginDescriptors);

      ToolingArtifactClassLoader toolingArtifactClassLoader = null;
      try {
        toolingArtifactClassLoader = builder.build();
        return action.call(toolingArtifactClassLoader.getArtifactPluginClassLoaders());
      } catch (Exception e) {
        throw new MojoFailureException("Error while generating class loaders in order to load the Extension Model", e);
      } finally {
        if (toolingArtifactClassLoader != null) {
          toolingArtifactClassLoader.dispose();
        }
      }
    } finally {
      deleteQuietly(applicationFolder);
    }
  }

  private DeployableArtifactClassLoaderFactory<ApplicationDescriptor> newTemporaryArtifactClassLoaderFactory() {
    return (String artifactId, ArtifactClassLoader parent, ApplicationDescriptor descriptor,
            List<ArtifactClassLoader> artifactPluginClassLoaders) -> new MuleDeployableArtifactClassLoader(artifactId,
                                                                                                           descriptor,
                                                                                                           descriptor
                                                                                                               .getClassLoaderModel()
                                                                                                               .getUrls(),
                                                                                                           parent
                                                                                                               .getClassLoader(),
                                                                                                           parent
                                                                                                               .getClassLoaderLookupPolicy(),
                                                                                                           artifactPluginClassLoaders);
  }

  private void createPomFile(Artifact artifact, String uuid, File applicationFolder) {
    Model model = new Model();
    model.setGroupId(uuid);
    model.setArtifactId(uuid);
    model.setVersion(getProductVersion());
    model.setPackaging("mule-application");
    model.setModelVersion("4.0.0");

    Dependency dependency = new Dependency();
    dependency.setGroupId(artifact.getGroupId());
    dependency.setArtifactId(artifact.getArtifactId());
    dependency.setVersion(artifact.getVersion());
    dependency.setClassifier(MULE_PLUGIN_CLASSIFIER);
    dependency.setType("jar");
    model.getDependencies().add(dependency);

    createDeployablePomFile(applicationFolder, model);
    createDeployablePomProperties(applicationFolder, model);
  }

  @FunctionalInterface
  interface Action<T> {

    T call(List<ArtifactClassLoader> artifactPluginClassLoaders) throws MojoFailureException, MojoExecutionException;
  }

}
