/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extension.maven;

import static java.lang.Boolean.parseBoolean;
import static java.lang.String.format;
import static org.apache.maven.plugins.annotations.LifecyclePhase.VERIFY;
import static org.apache.maven.plugins.annotations.ResolutionScope.TEST;
import static org.mule.extension.maven.ExtensionModelMojo.TEMPORAL_EXTENSION_MODEL_JSON;
import org.mule.extension.maven.documentation.DocumentationGenerator;
import org.mule.plugin.maven.AbstractMuleMojo;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.extension.api.persistence.ExtensionModelJsonSerializer;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Files;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectHelper;

/**
 * A simple maven plugin that generates Asciidoc and HTML documentation pages for an extension.
 */
@Mojo(name = "documentation", defaultPhase = VERIFY, requiresDependencyResolution = TEST, threadSafe = true)
public class ExtensionDocumentationMojo extends AbstractMuleMojo {

  private static final String DOC_SUFFIX = "-documentation";
  private static final String ZIP = "zip";

  /**
   * The maven project where this MOJO is being executed.
   */
  @Parameter(defaultValue = "${project}", required = true)
  private MavenProject project;

  /**
   * The output directory where the temporal generated documentation files are going to be located.
   */
  @Parameter(defaultValue = "${project.build.directory}/docs", required = true)
  private File outputDir;

  /**
   * Whether the generation of the docs should be skipped or not
   */
  @Parameter(defaultValue = "false", property = "skipDocumentation")
  private Boolean skipDocumentation;

  @Component
  private MavenProjectHelper projectHelper;

  @Override
  public void execute() throws MojoExecutionException, MojoFailureException {
    if (parseBoolean(System.getProperty("mule.maven.extension.model.disable", "false")) || skipDocumentation) {
      getLog().info("Extension Documentation generation skipped.");
      return;
    }
    if (!outputDir.exists()) {
      outputDir.mkdirs();
    }
    ExtensionModel extension = getExtensionModel();
    DocumentationGenerator documentationGenerator = new DocumentationGenerator(extension, outputDirectory);
    String asciidoc = documentationGenerator.generateAsciidoc(extension);
    String html = documentationGenerator.generateHtml(asciidoc);
    generateFile(getSanitizedName(extension) + DOC_SUFFIX + ".adoc", asciidoc);
    generateFile(getSanitizedName(extension) + DOC_SUFFIX + ".html", html);
  }

  /**
   * As the current MOJO ({@link ExtensionDocumentationMojo}) is executed after {@link ExtensionModelMojo}, instead of
   * regenerating the {@link ExtensionModel} (and its dependencies) we will deserialize it from what's has been left over in the
   * previous MOJO.
   *
   * @return the already generated {@link ExtensionModel}
   * @throws MojoExecutionException if the serialized model fails to be read
   */
  private ExtensionModel getExtensionModel() throws MojoExecutionException {
    final File generatedExtensionModelFile = new File(outputDirectory, TEMPORAL_EXTENSION_MODEL_JSON);
    try {
      final String serializedExModel = new String(Files.readAllBytes(generatedExtensionModelFile.toPath()));
      return new ExtensionModelJsonSerializer(true).deserialize(serializedExModel);
    } catch (IOException e) {
      throw new MojoExecutionException(
                                       format("Cannot obtain/read the file [%s] to deserialize the ExtensionModel",
                                              generatedExtensionModelFile.getAbsolutePath()),
                                       e);
    }
  }

  private String getSanitizedName(ExtensionModel extensionModel) {
    return extensionModel.getName().toLowerCase().replace(" ", "-");
  }

  private void generateFile(String fileName, String content) throws MojoExecutionException {
    File file = new File(outputDir, fileName);
    try {
      PrintWriter printWriter = new PrintWriter(file);
      printWriter.print(content);
      printWriter.close();
    } catch (FileNotFoundException e) {
      throw new MojoExecutionException(
                                       format("Cannot write the file [%s] (it does not exist), cannot generate documentation",
                                              file.getAbsolutePath()),
                                       e);
    }
    getLog().info(format("Generated file [%s]", file.getAbsolutePath()));
  }
}
