/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.extension.maven.loader;

import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.container.api.ContainerClassLoaderProvider.createContainerClassLoader;
import static org.mule.runtime.container.api.ModuleRepository.createModuleRepository;
import static org.mule.runtime.core.api.config.bootstrap.ArtifactType.PLUGIN;
import static org.mule.runtime.core.internal.util.JarUtils.loadFileContentFrom;
import static org.mule.runtime.deployment.model.api.artifact.ArtifactDescriptorFactoryProvider.artifactDescriptorFactoryProvider;
import static org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptor.MULE_ARTIFACT_JSON_DESCRIPTOR;
import static org.mule.runtime.module.artifact.api.descriptor.ArtifactPluginDescriptor.MULE_ARTIFACT_PATH_INSIDE_JAR;

import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import static com.google.common.io.Files.createTempDir;

import org.mule.extension.maven.ExtensionDescriptorMojo;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptor;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptorBuilder;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.deployment.meta.Product;
import org.mule.runtime.api.deployment.persistence.AbstractMuleArtifactModelJsonSerializer;
import org.mule.runtime.api.deployment.persistence.MulePluginModelJsonSerializer;
import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.runtime.deployment.model.api.artifact.DescriptorLoaderRepositoryFactory;
import org.mule.runtime.deployment.model.api.plugin.resolver.PluginDependenciesResolver;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.AbstractArtifactDescriptorFactory;
import org.mule.runtime.module.artifact.api.descriptor.ApplicationDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptorCreateException;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptorValidatorBuilder;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactPluginDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.DescriptorLoaderRepository;
import org.mule.runtime.module.artifact.api.plugin.LoaderDescriber;
import org.mule.runtime.module.deployment.impl.internal.plugin.PluginExtendedBundleDescriptorAttributes;
import org.mule.runtime.module.deployment.impl.internal.plugin.PluginExtendedClassLoaderConfigurationAttributes;
import org.mule.runtime.module.deployment.impl.internal.plugin.PluginExtendedDeploymentProperties;

import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.function.Predicate;

import org.apache.commons.io.FileUtils;

public class MuleArtifactResourcesRegistry {

  private final AbstractArtifactDescriptorFactory<MuleApplicationModel, ApplicationDescriptor> applicationDescriptorFactory;
  private final PluginDependenciesResolver pluginDependenciesResolver;
  private final ArtifactClassLoader containerArtifactClassLoader;

  public MuleArtifactResourcesRegistry(Predicate<PluginDescriptor> enableXmlValidations) {
    DescriptorLoaderRepository descriptorLoaderRepository =
        new DescriptorLoaderRepositoryFactory().createDescriptorLoaderRepository();
    ModuleRepository moduleRepository = createModuleRepository(this.getClass().getClassLoader(), createTempDir());

    ArtifactDescriptorValidatorBuilder artifactDescriptorValidatorBuilder = ArtifactDescriptorValidatorBuilder.builder();
    AbstractArtifactDescriptorFactory<MulePluginModel, ArtifactPluginDescriptor> artifactPluginDescriptorFactory =
        new ArtifactPluginDescriptorFactoryExtension(descriptorLoaderRepository, artifactDescriptorValidatorBuilder,
                                                     enableXmlValidations);

    applicationDescriptorFactory = artifactDescriptorFactoryProvider()
        .createApplicationDescriptorFactory(artifactPluginDescriptorFactory, descriptorLoaderRepository,
                                            artifactDescriptorValidatorBuilder);
    pluginDependenciesResolver =
        artifactDescriptorFactoryProvider().createBundlePluginDependenciesResolver(artifactPluginDescriptorFactory);

    containerArtifactClassLoader = createContainerClassLoader(moduleRepository, this.getClass().getClassLoader());
  }

  public AbstractArtifactDescriptorFactory<MuleApplicationModel, ApplicationDescriptor> getApplicationDescriptorFactory() {
    return applicationDescriptorFactory;
  }

  public PluginDependenciesResolver getPluginDependenciesResolver() {
    return pluginDependenciesResolver;
  }

  public ArtifactClassLoader getContainerArtifactClassLoader() {
    return containerArtifactClassLoader;
  }

  public static class PluginDescriptor {

    private final File pluginFile;
    private final MuleArtifactLoaderDescriptor muleArtifactLoaderDescriptor;

    public PluginDescriptor(File pluginFile, MuleArtifactLoaderDescriptor muleArtifactLoaderDescriptor) {
      this.pluginFile = pluginFile;
      this.muleArtifactLoaderDescriptor = muleArtifactLoaderDescriptor;
    }

    public File getPluginFile() {
      return pluginFile;
    }

    public MuleArtifactLoaderDescriptor getMuleArtifactLoaderDescriptor() {
      return muleArtifactLoaderDescriptor;
    }
  }

  private final class ArtifactPluginDescriptorFactoryExtension
      extends AbstractArtifactDescriptorFactory<MulePluginModel, ArtifactPluginDescriptor> {

    private final Predicate<PluginDescriptor> enableXmlValidations;

    private ArtifactPluginDescriptorFactoryExtension(DescriptorLoaderRepository descriptorLoaderRepository,
                                                     ArtifactDescriptorValidatorBuilder artifactDescriptorValidatorBuilder,
                                                     Predicate<PluginDescriptor> enableXmlValidations) {
      super(descriptorLoaderRepository, artifactDescriptorValidatorBuilder);
      this.enableXmlValidations = enableXmlValidations;
    }

    /**
     * Custom implementation that won't fail if the pluginFile is not a jar/zip file. Works with target/classes too.
     */
    @Override
    public ArtifactPluginDescriptor create(File pluginFile, Optional<Properties> deploymentProperties) {
      try {
        Optional<byte[]> jsonDescriptorContentOptional = empty();
        String mulePluginJsonPath = MULE_ARTIFACT_PATH_INSIDE_JAR + "/" + MULE_ARTIFACT_JSON_DESCRIPTOR;
        jsonDescriptorContentOptional =
            getMuleArtifactJsonDescriptorContent(pluginFile, jsonDescriptorContentOptional, mulePluginJsonPath);

        MulePluginModel artifactModel = jsonDescriptorContentOptional
            .map(jsonDescriptorContent -> loadModelFromJson(new String(jsonDescriptorContent)))
            .orElseThrow(() -> new ArtifactDescriptorCreateException(format("The plugin descriptor '%s' on plugin file '%s' is not present",
                                                                            mulePluginJsonPath, pluginFile)));

        // Create another plugin model to set as required product MULE and minMuleVersion 4.1.1 due to validations cannot be
        // disabled
        MulePluginModel.MulePluginModelBuilder mulePluginModelBuilder = new MulePluginModel.MulePluginModelBuilder();
        mulePluginModelBuilder
            .setName(artifactModel.getName())
            .setMinMuleVersion("4.1.1")
            .setRequiredProduct(Product.MULE)
            .withBundleDescriptorLoader(artifactModel.getBundleDescriptorLoader())
            .withClassLoaderModelDescriptorLoader(artifactModel.getClassLoaderModelLoaderDescriptor());

        artifactModel.getLicense().ifPresent(licenseModel -> mulePluginModelBuilder.withLicenseModel()
            .setAllowsEvaluationLicense(licenseModel.isAllowsEvaluation())
            .setProvider(licenseModel.getProvider())
            .setRequiredEntitlement(licenseModel.getRequiredEntitlement().orElse(null)));

        artifactModel.getExtensionModelLoaderDescriptor().ifPresent(extensionModelDescriptorLoader -> {
          MuleArtifactLoaderDescriptorBuilder muleArtifactLoaderDescriptorBuilder =
              mulePluginModelBuilder.withExtensionModelDescriber();
          muleArtifactLoaderDescriptorBuilder.setId(extensionModelDescriptorLoader.getId());
          extensionModelDescriptorLoader.getAttributes().entrySet().stream()
              .forEach(entry -> muleArtifactLoaderDescriptorBuilder.addProperty(entry.getKey(), entry.getValue()));

          // If the current plugin being built is an XML SDK we should always validate its XML
          if (artifactModel.getExtensionModelLoaderDescriptor().isPresent() &&
              enableXmlValidations
                  .test(new PluginDescriptor(pluginFile, artifactModel.getExtensionModelLoaderDescriptor().get()))) {
            muleArtifactLoaderDescriptorBuilder.addProperty(ExtensionDescriptorMojo.VALIDATE_XML, true);
          }
        });

        MulePluginModel pluginModel = mulePluginModelBuilder.build();
        return this.loadFromJsonDescriptor(pluginFile, pluginModel, deploymentProperties);
      } catch (ArtifactDescriptorCreateException e) {
        throw e;
      } catch (IOException e) {
        throw new ArtifactDescriptorCreateException(e);
      }
    }

    private Optional<byte[]> getMuleArtifactJsonDescriptorContent(File pluginFile,
                                                                  Optional<byte[]> jsonDescriptorContentOptional,
                                                                  String mulePluginJsonPath)
        throws IOException {
      if (pluginFile.isFile()) {
        checkArgument(pluginFile.getName().endsWith(".jar"),
                      "provided file is not a plugin: " + pluginFile.getAbsolutePath());
        jsonDescriptorContentOptional = loadFileContentFrom(pluginFile, mulePluginJsonPath);
      } else {
        File mulePluginJsonFile =
            pluginFile.toPath().resolve(Paths.get(MULE_ARTIFACT_PATH_INSIDE_JAR, MULE_ARTIFACT_JSON_DESCRIPTOR)).toFile();
        if (mulePluginJsonFile.exists()) {
          jsonDescriptorContentOptional = of(FileUtils.readFileToByteArray(mulePluginJsonFile));
        }
      }
      return jsonDescriptorContentOptional;
    }

    @Override
    protected Map<String, Object> getClassLoaderConfigurationAttributes(Optional<Properties> deploymentPropertiesOptional,
                                                                        MuleArtifactLoaderDescriptor classLoaderModelLoaderDescriptor,
                                                                        BundleDescriptor bundleDescriptor) {
      Map<String, Object> attributes =
          super.getClassLoaderConfigurationAttributes(deploymentPropertiesOptional, classLoaderModelLoaderDescriptor,
                                                      bundleDescriptor);

      if (deploymentPropertiesOptional.isPresent()) {
        Properties deploymentProperties = deploymentPropertiesOptional.get();
        if (deploymentProperties instanceof PluginExtendedDeploymentProperties) {
          PluginExtendedDeploymentProperties pluginExtendedDeploymentProperties =
              (PluginExtendedDeploymentProperties) deploymentProperties;
          return new PluginExtendedClassLoaderConfigurationAttributes(attributes,
                                                                      pluginExtendedDeploymentProperties
                                                                          .getDeployableArtifactDescriptor());
        }
      }
      return attributes;
    }

    @Override
    protected Map<String, Object> getBundleDescriptorAttributes(MuleArtifactLoaderDescriptor bundleDescriptorLoader,
                                                                Optional<Properties> deploymentPropertiesOptional) {
      Map<String, Object> attributes =
          super.getBundleDescriptorAttributes(bundleDescriptorLoader, deploymentPropertiesOptional);

      if (deploymentPropertiesOptional.isPresent()) {
        Properties deploymentProperties = deploymentPropertiesOptional.get();
        if (deploymentProperties instanceof PluginExtendedDeploymentProperties) {
          PluginExtendedDeploymentProperties pluginExtendedDeploymentProperties =
              (PluginExtendedDeploymentProperties) deploymentProperties;
          return new PluginExtendedBundleDescriptorAttributes(attributes,
                                                              pluginExtendedDeploymentProperties.getPluginBundleDescriptor());
        }
      }
      return attributes;
    }

    @Override
    protected ArtifactType getArtifactType() {
      return PLUGIN;
    }

    @Override
    protected AbstractMuleArtifactModelJsonSerializer<MulePluginModel> getMuleArtifactModelJsonSerializer() {
      return new MulePluginModelJsonSerializer();
    }

    @Override
    protected void doDescriptorConfig(MulePluginModel artifactModel, ArtifactPluginDescriptor descriptor, File artifactLocation) {
      artifactModel.getExtensionModelLoaderDescriptor().ifPresent(extensionModelDescriptor -> {
        final LoaderDescriber loaderDescriber = new LoaderDescriber(extensionModelDescriptor.getId());
        loaderDescriber.addAttributes(extensionModelDescriptor.getAttributes());
        descriptor.setExtensionModelDescriptorProperty(loaderDescriber);
      });
      artifactModel.getLicense().ifPresent(descriptor::setLicenseModel);
    }

    @Override
    protected ArtifactPluginDescriptor createArtifactDescriptor(File artifactLocation, String name,
                                                                Optional<Properties> deploymentProperties) {
      return new ArtifactPluginDescriptor(name);
    }
  }

}
