/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extension.maven;

import static java.io.File.separator;
import static java.lang.String.format;
import static java.nio.file.Files.copy;
import static org.apache.maven.plugins.annotations.LifecyclePhase.PACKAGE;
import static org.apache.maven.plugins.annotations.ResolutionScope.RUNTIME;
import static org.mule.extension.maven.ExtensionDescriptorMojo.descriptorPathOrFail;

import org.mule.plugin.maven.AbstractPackagePluginMojo;
import org.mule.plugin.maven.ModuleArchiver;

import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.codehaus.plexus.archiver.ArchiverException;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.nio.file.Paths;


/**
 * Build a Mule extension archive.
 */
@Mojo(name = "package", defaultPhase = PACKAGE, requiresDependencyResolution = RUNTIME, threadSafe = true)
public class ExtensionPackageMojo extends AbstractPackagePluginMojo {

  private static final String ICON_SVG = "icon.svg";
  private static final String GENERIC_CONNECTOR_SVG = "generic-connector.svg";

  private static String ICON_FOLDER = separator + "icon" + separator;
  /**
   * The SVG icon will be extracted from the relative location described below from the current base project, if not found a
   * default one will be used.
   */
  private static final String EXPECTED_ICON_RELATIVE_PATH = ICON_FOLDER + ICON_SVG;

  @Override
  protected void addToArchiver(ModuleArchiver archiver) throws MojoFailureException {
    super.addToArchiver(archiver);
    addClasses(archiver);
    addPluginDescriptor(archiver);
    addIcons(archiver);
  }

  /**
   * Looks for the icons under {@link #EXPECTED_ICON_RELATIVE_PATH} and copies it into the {@code archiver}. If the main icon is
   * missing in the plugin's folder, it will copy a default one. Any other icon will be also copied.
   *
   * @param archiver to store all the icons
   * @throws MojoFailureException if there's an issue while copying the icon
   */
  private void addIcons(ModuleArchiver archiver) throws MojoFailureException {
    File iconFile = new File(project.getBasedir(), EXPECTED_ICON_RELATIVE_PATH);
    if (!iconFile.exists()) {
      getLog()
          .warn(format("The icon [%s] was not found in the current project, a generic one will be used instead. To replace it with a custom icon put an SVG file under [%s]",
                       iconFile.getAbsolutePath(), EXPECTED_ICON_RELATIVE_PATH));
      try (InputStream resource = getClass().getClassLoader().getResourceAsStream(GENERIC_CONNECTOR_SVG)) {
        iconFile = new File(outputDirectory, ICON_SVG);
        if (!iconFile.exists()) {
          try {
            copy(resource, Paths.get(iconFile.getAbsolutePath()));
          } catch (IOException e) {
            throw new MojoFailureException(format("There was an issue while copying the resource [%s]", GENERIC_CONNECTOR_SVG),
                                           e);
          }
        }
      } catch (IOException e) {
        throw new MojoFailureException(e.getMessage(), e);
      }
      archiver.addFile(iconFile, RESERVED_FOLDER_MULE_ARTIFACT + ICON_SVG);
    }
    File iconsFolder = new File(project.getBasedir(), ICON_FOLDER);

    File[] files = iconsFolder.listFiles((dir, name) -> name.toLowerCase().endsWith(".svg"));
    if (files != null) {
      for (File anIcon : files) {
        archiver.addFile(anIcon, RESERVED_FOLDER_MULE_ARTIFACT + anIcon.getName());
      }
    }
  }

  /**
   * Adds the {@link #MULE_ARTIFACT_JSON} file, extracted from the current extensions JAR into the {@link #META_INF} folder.
   *
   * @param archiver to store the current {@link #MULE_ARTIFACT_JSON} file
   * @throws MojoFailureException
   */
  private void addPluginDescriptor(ModuleArchiver archiver) throws MojoFailureException {
    final Path path = descriptorPathOrFail(outputDirectory);
    archiver.addFile(path.toFile(), MULE_PLUGIN_JSON_JAR_DESTINATION);
  }

  /**
   * Does nothing, plugins do not package their dependencies
   *
   * @param archiver to store all dependencies
   */
  @Override
  protected void addDependencies(ModuleArchiver archiver) {
    // Do nothing, plugins do not package their dependencies
  }

  @Override
  protected String getArtifactClassifier() {
    return MULE_PLUGIN_CLASSIFIER;
  }

  /**
   * Adds no plugin.properties as extensions will have an internal "maven-like" repository that will work directly with the
   * mule-artifact.json this there's no need to populate the {@code pluginPropertiesFile}.
   *
   * @see #addPluginDescriptor(ModuleArchiver)
   *
   *      TODO(fernandezlautaro): MULE-11136 refactor this object (and its superclasses) when working in this jira (every plugin
   *      will have mule-artifact.json file)
   *
   * @param pluginPropertiesFile
   * @throws ArchiverException
   */
  @Override
  protected void doAddArtifactProperties(File pluginPropertiesFile) throws MojoFailureException {
    // Does nothing
  }

  /**
   * Adds no plugin.properties as extensions will have an internal "maven-like" repository that will work directly with the
   * mule-artifact.json
   *
   * @see #addPluginDescriptor(ModuleArchiver)
   *
   *      TODO(fernandezlautaro): MULE-11136 refactor this object (and its superclasses) when working in this jira (every plugin
   *      will have mule-arti.json file)
   *
   * @param archiver
   * @throws ArchiverException
   */
  @Override
  protected void addArtifactProperties(ModuleArchiver archiver) throws ArchiverException {
    // Does nothing
  }

  @Override
  protected File getMuleArtifactFile() {
    return new File(this.outputDirectory, this.finalName + "-" + MULE_PLUGIN_CLASSIFIER + ".jar");
  }
}
