/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extensions.maven.shade;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import static java.lang.Long.MIN_VALUE;
import static java.nio.charset.Charset.defaultCharset;
import static org.mule.extensions.maven.shade.ExtensionShadingUtils.shadeClassNames;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;

import org.apache.commons.io.IOUtils;
import org.apache.maven.plugins.shade.mojo.PackageRelocation;
import org.apache.maven.plugins.shade.relocation.Relocator;
import org.apache.maven.plugins.shade.resource.ReproducibleResourceTransformer;
import org.codehaus.plexus.util.IOUtil;

public class MuleArtifactDescriptorTransformer implements ReproducibleResourceTransformer {

  private static final String ARTIFACT_JSON_PATH = "META-INF/mule-artifact/mule-artifact.json";

  private long time = MIN_VALUE;
  private final List<PackageRelocation> relocations;
  private String shadedArtifactJson;

  public MuleArtifactDescriptorTransformer(List<PackageRelocation> relocations) {
    this.relocations = relocations;
  }

  @Override
  public boolean canTransformResource(String resource) {
    return ARTIFACT_JSON_PATH.equals(resource);
  }

  @Override
  public void processResource(String resource, InputStream is, List<Relocator> relocators) throws IOException {
    processResource(resource, is, relocators, time);
  }

  @Override
  public void processResource(String resource, InputStream is, List<Relocator> relocators, long time) throws IOException {
    if (time > this.time) {
      shadedArtifactJson = shadeClassNames(IOUtils.toString(is, defaultCharset()), relocations);
      this.time = time;
    }
  }

  @Override
  public boolean hasTransformedResource() {
    return true;
  }

  @Override
  public void modifyOutputStream(JarOutputStream os) throws IOException {
    JarEntry entry = new JarEntry(ARTIFACT_JSON_PATH);
    entry.setTime(time);

    os.putNextEntry(entry);
    IOUtil.copy(new ByteArrayInputStream(shadedArtifactJson.getBytes()), os);
  }

  public String getShadedArtifactJson() {
    return shadedArtifactJson;
  }
}
