/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extensions.maven.shade;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import static java.lang.String.format;
import static java.nio.charset.Charset.defaultCharset;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableList;
import static org.apache.commons.io.FileUtils.readFileToString;
import static org.apache.commons.io.FileUtils.writeStringToFile;
import static org.apache.maven.plugins.annotations.LifecyclePhase.PACKAGE;
import static org.apache.maven.plugins.annotations.ResolutionScope.RUNTIME;
import static org.mule.extensions.maven.shade.ExtensionShadingUtils.shadeClassNames;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.List;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.shade.mojo.PackageRelocation;
import org.apache.maven.plugins.shade.mojo.ShadeMojo;
import org.apache.maven.plugins.shade.resource.ResourceTransformer;

/**
 * Mojo that performs shading delegating to the Shader component.
 *
 * @author Jason van Zyl
 * @author Mauro Talevi
 * @author David Blevins
 * @author Hiram Chirino
 */
@Mojo(name = "shade", defaultPhase = PACKAGE, threadSafe = true, requiresDependencyResolution = RUNTIME)
public class ExtensionShadeMojo extends ShadeMojo {

  @Override
  public void execute() throws MojoExecutionException {
    if (skipShade) {
      return;
    }
    final File originalArtifactFile = project.getArtifact().getFile();
    File rootDirectory = originalArtifactFile.getParentFile();
    List<PackageRelocation> relocations = getRelocations();

    if (originalArtifactFile.isDirectory()) {
      String[] pluginFileNames = rootDirectory.list(new FilenameFilter() {

        @Override
        public boolean accept(File dir, String name) {
          return name.startsWith(project.getArtifactId()) && name.endsWith("-mule-plugin.jar");
        }
      });
      if (pluginFileNames.length > 0) {
        project.getArtifact().setFile(new File(rootDirectory, pluginFileNames[0]));
      }

      shadeTempFile(rootDirectory, "temporal-extension-model.json", relocations);
      shadeTempFile(rootDirectory, "classes/META-INF/mule-artifact/mule-artifact.json", relocations);
    }

    if (transformers != null) {
      ResourceTransformer[] enhancedTransformers = new ResourceTransformer[transformers.length + 1];
      System.arraycopy(transformers, 0, enhancedTransformers, 0, transformers.length);
      enhancedTransformers[transformers.length] = new MuleArtifactDescriptorTransformer(relocations);
      transformers = enhancedTransformers;
    } else {
      transformers = new ResourceTransformer[1];
      transformers[0] = new MuleArtifactDescriptorTransformer(relocations);
    }

    super.execute();

    //Reset artifact file to its original state.
    project.getArtifact().setFile(originalArtifactFile);
  }

  private void shadeTempFile(File rootDirectory, String filePath, List<PackageRelocation> relocations) {
    File tempFile = Paths.get(rootDirectory.toString(), filePath).toFile();
    if (!tempFile.exists()) {
      getLog().warn(format("Couldn't find temporal file at '%s'. Will not shade it", tempFile.getAbsolutePath()));
      return;
    }

    String tempFileContent;
    try {
      tempFileContent = readFileToString(tempFile, defaultCharset());
    } catch (IOException e) {
      getLog().warn(format("Couldn't read temp file at '%s'. Will not shade it", tempFile.getAbsolutePath()), e);
      return;
    }

    try {
      writeStringToFile(tempFile, shadeClassNames(tempFileContent, relocations), defaultCharset(), false);
    } catch (IOException e) {
      getLog().warn(format("Couldn't overwrite shaded temp file at '%s'. Will not shade it", tempFile.getAbsolutePath()), e);
    }
  }

  private List<PackageRelocation> getRelocations() {
    if (relocations != null) {
      return unmodifiableList(asList(relocations));
    }

    return emptyList();
  }
}
