/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.plugin.maven;

import static java.io.File.separator;
import static java.lang.String.format;
import static java.lang.String.join;
import static java.nio.file.FileVisitResult.CONTINUE;
import static java.nio.file.Files.walkFileTree;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.maven.artifact.Artifact.SCOPE_COMPILE;
import static org.apache.maven.artifact.Artifact.SCOPE_PROVIDED;
import static org.apache.maven.artifact.Artifact.SCOPE_RUNTIME;
import static org.apache.maven.artifact.Artifact.SCOPE_SYSTEM;

import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.List;
import java.util.Set;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.model.Dependency;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProjectHelper;
import org.codehaus.plexus.archiver.ArchiverException;

public abstract class AbstractPackagePluginMojo extends AbstractMuleMojo {

  public static final String MULE_PLUGIN_CLASSIFIER = "mule-plugin";

  static final String PLUGIN_PROPERTIES_FILE_NAME = "plugin.properties";

  private static final String BUNDLE_DESCRIPTOR_SEPARATOR = ":";
  private static final String CLASSES = "classes";
  private static final String CE_RUNTIME_GROUP_ID = "org.mule.runtime";
  private static final String EE_RUNTIME_GROUP_ID = "com.mulesoft.mule.runtime";
  private static final List<String> forbiddenScopesRuntimeDependencies = asList(SCOPE_COMPILE, SCOPE_RUNTIME, SCOPE_SYSTEM);

  /**
   * As the packaging of a plugin is defined as a JAR, all the compiled classes must be at the root level so that contributing to
   * the classpath is just pointing to the JAR file, rather than (a) extracting the classes in a specific folder to point that URL
   * or (b) building a custom plugin to consume classes.
   */
  private static final String RESERVED_DESTINATION_CLASSES = "";

  protected abstract void doAddArtifactProperties(File pluginPropertiesFile) throws MojoFailureException;

  @Component
  private MavenProjectHelper projectHelper;
  /**
   * List of exclusion elements (having groupId and artifactId children) to exclude from the application archive.
   */
  @Parameter
  private List<Exclusion> exclusions;
  /**
   * List of inclusion elements (having groupId and artifactId children) to include from the application archive.
   */
  @Parameter
  private List<Inclusion> inclusions;

  public AbstractPackagePluginMojo() {
    super();
  }

  @Override
  public final void execute() throws MojoExecutionException, MojoFailureException {
    initialize();
    validateDependencies();
    doExecute();
    File artifactFile = getMuleArtifactFile();
    try {
      createMulePlugin(artifactFile);
    } catch (ArchiverException e) {
      throw new MojoExecutionException("Exception creating the Mule Artifact", e);
    }

    this.projectHelper.attachArtifact(this.project, getArtifactType(), getArtifactClassifier(), artifactFile);
  }

  /**
   * Goes over all dependencies in the current project and for anyone that's a runtime dependency will assert it's not declared as
   * any of the scopes defined in {@link #forbiddenScopesRuntimeDependencies}.
   * <p/>
   * Any runtime dependency is discovered as it has a group ID starting with {@link #CE_RUNTIME_GROUP_ID} or
   * {@link #EE_RUNTIME_GROUP_ID}.
   *
   * @throws MojoFailureException if there is at least one runtime dependency consumed with any of the defined forbidden scopes.
   */
  private void validateDependencies() throws MojoFailureException {
    final List<String> forbiddenRuntimeDependencies = project.getDependencies().stream()
        .filter(dependency -> forbiddenScopesRuntimeDependencies.contains(dependency.getScope()))
        .filter(dependency -> dependency.getGroupId().startsWith(CE_RUNTIME_GROUP_ID)
            || dependency.getGroupId().startsWith(EE_RUNTIME_GROUP_ID))
        .map(Dependency::toString)
        .collect(toList());
    if (!forbiddenRuntimeDependencies.isEmpty()) {
      throw new MojoFailureException(format("There are [%d] runtime dependencies (groupId matches either [%s] or [%s]) which <scope> must be [%s], but found [%s]. Full list:\n %s",
                                            forbiddenRuntimeDependencies.size(),
                                            CE_RUNTIME_GROUP_ID,
                                            EE_RUNTIME_GROUP_ID,
                                            SCOPE_PROVIDED,
                                            join(", ", forbiddenScopesRuntimeDependencies),
                                            join(",\n", forbiddenRuntimeDependencies)));
    }
  }

  protected String getArtifactType() {
    // TODO(fernandezlautaro): MULE-11383 all artifacts must be .jar files, when done remove this method and hardcode jar
    return "zip";
  }

  protected String getArtifactClassifier() {
    return EMPTY;
  }

  protected void createMulePlugin(final File plugin) throws MojoExecutionException, MojoFailureException, ArchiverException {
    ModuleArchiver archiver = new ModuleArchiver();
    addToArchiver(archiver);

    archiver.setDestFile(plugin);

    try {
      plugin.delete();
      archiver.createArchive();
    } catch (IOException e) {
      getLog().error("Cannot create archive", e);
    }
  }

  protected void addToArchiver(ModuleArchiver archiver) throws MojoFailureException {
    addDependencies(archiver);
    addArtifactProperties(archiver);
    addPOMFile(archiver);
  }

  /**
   * Adds the pom file for this project into META-INF/maven/${groupId}/${artifactId}/.
   *
   * @param archiver to store the current dependency file
   */
  protected void addPOMFile(ModuleArchiver archiver) {
    archiver.addFile(project.getFile(),
                     Paths.get(META_INF, "maven", project.getGroupId(), project.getArtifactId(), "pom.xml").toString());
  }

  protected void addArtifactProperties(ModuleArchiver moduleArchiver) throws MojoFailureException {
    File pluginPropertiesFile = new File(outputDirectory, CLASSES + separator + PLUGIN_PROPERTIES_FILE_NAME);
    if (pluginPropertiesFile.exists()) {
      doAddArtifactProperties(pluginPropertiesFile);
      moduleArchiver.add(pluginPropertiesFile);
    }
  }

  protected void addDependencies(ModuleArchiver archiver) throws ArchiverException {
    for (Artifact artifact : getArtifactsToArchive()) {
      getLog().info(format("Adding <%1s> as a lib", artifact.getId()));
      archiver.addLib(artifact.getFile());
    }
  }

  private Set<Artifact> getArtifactsToArchive() {
    ArtifactFilter filter = new ArtifactFilter(project, inclusions, exclusions);
    return filter.getArtifactsToArchive();
  }

  protected Set<String> getPluginDependencies() {
    List<Dependency> allDependencies = project.getDependencies();
    return allDependencies.stream()
        .filter(d -> d.getClassifier() != null && d.getClassifier().equals(MULE_PLUGIN_CLASSIFIER))
        .map(d -> d.getGroupId() + BUNDLE_DESCRIPTOR_SEPARATOR + d.getArtifactId() + BUNDLE_DESCRIPTOR_SEPARATOR + d.getVersion())
        .collect(toSet());
  }

  /**
   * Method to give the capability of initialize implementations of this class as the first action of the
   * {@link AbstractPackagePluginMojo#execute()}
   *
   * @throws MojoFailureException when an initialization failure occurs
   * @throws MojoExecutionException when an initialization error occurs
   */
  protected void initialize() throws MojoFailureException, MojoExecutionException {}

  /**
   * Method to give the capability of contribute with the {@link AbstractPackagePluginMojo#execute()
   *
   * @throws MojoFailureException when an execution failure occurs
   * @throws MojoExecutionException when an execution error occurs
   */
  protected void doExecute() throws MojoFailureException, MojoExecutionException {}

  /**
   * Copies the complete files encountered in {@link #outputDirectory}/{@code folder}, into the current {@code archiver}
   *
   * @param archiver to store all dependencies
   * @param destination folder in which all found files will be stored inside of the archive file
   * @param startPath path to introspect
   * @throws MojoFailureException if there's an issue while working with the files under {@link #outputDirectory}/{@code folder}
   */
  protected void walkWhileCopyingFolder(final ModuleArchiver archiver, final String destination, final Path startPath) {
    try {
      walkFileTree(startPath, new SimpleFileVisitor<Path>() {

        @Override
        public FileVisitResult visitFile(Path currentPath, BasicFileAttributes attrs) throws IOException {
          final String dest = destination + startPath.relativize(currentPath);
          archiver.addFile(currentPath.toFile(), dest);
          return CONTINUE;
        }
      });
    } catch (IOException e) {
      throw new ArchiverException(format("Error copying folder [%s], from start [%s]", destination, startPath.toString()), e);
    }
  }

  /**
   * Adds the complete structure under {@link #outputDirectory}/{@link #CLASSES} to the {@code archiver}
   *
   * @param archiver to store the {@link #CLASSES} structure
   */
  protected void addClasses(ModuleArchiver archiver) {
    final Path startPath = Paths.get(outputDirectory.getAbsolutePath(), CLASSES);
    walkWhileCopyingFolder(archiver, RESERVED_DESTINATION_CLASSES, startPath);
  }
}
