/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.plugin.maven;

import java.util.List;
import java.util.function.Predicate;

import org.apache.maven.artifact.Artifact;

/**
 * Declares libs that must be used only with a minimal version of Java when used in runtime.
 * 
 * @since 1.6
 */
public class MultiJavaRelease {

  /**
   * The version of java starting on which the libraries matched by `includes` will be used. For previous java versions, those
   * matched libraries won't be used.
   */
  private int javaVersion;

  /**
   * The pattern to include, in format {@code groupId:artifactId}
   */
  private List<String> includes;

  public int getJavaVersion() {
    return javaVersion;
  }

  public List<String> getIncludes() {
    return includes;
  }

  public Predicate<Artifact> getIncludesMatcher() {
    return artifact -> {
      return includes.stream()
          .anyMatch(include -> {
            final String[] splitInclude = include.split(":");
            String includeGroupId = splitInclude[0];
            String includeArtifactId = splitInclude[1];

            if ((includeGroupId.equals("*")
                || includeGroupId.equals(artifact.getGroupId()))
                && (includeArtifactId.equals("*")
                    || includeArtifactId.equals(artifact.getArtifactId()))) {
              return true;
            }

            return false;
          });
    };
  }

  @Override
  public String toString() {
    return this.getClass().getName() + "{javaVersion: " + javaVersion + "; includes: " + includes + "}";
  }
}
