/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.plugin.maven;

import static java.io.File.separator;
import static java.lang.String.format;
import static java.nio.file.Files.notExists;

import static org.apache.maven.plugins.annotations.LifecyclePhase.PACKAGE;
import static org.apache.maven.plugins.annotations.ResolutionScope.RUNTIME;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;

import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;


/**
 * Build a Mule service archive.
 */
@Mojo(name = "package-service", defaultPhase = PACKAGE, requiresDependencyResolution = RUNTIME, threadSafe = true)
public class ServicePackageMojo extends AbstractPackagePluginMojo {

  private static final String MULE_SERVICE = "mule-service";
  @Parameter(defaultValue = "${project.basedir}", readonly = true)
  private File baseDir;

  @Parameter
  protected String serviceProviderClassName;

  @Override
  protected void doAddArtifactProperties(File pluginPropertiesFile) throws MojoFailureException {
    // Do nothing
  }

  /**
   * Returns a {@link Path} to an existing descriptor file or fails.
   *
   * @param outputDirectory to look for the current output directory
   * @return an existing {@link Path} to a descriptor file
   * @throws MojoFailureException if the descriptor file is absent (probably because this Mojo hasn't been executed)
   */
  public static Path descriptorPathOrFail(File outputDirectory) throws MojoFailureException {
    final Path path = Paths.get(outputDirectory.getAbsolutePath(), "classes" + separator + MULE_PLUGIN_JSON_JAR_DESTINATION);
    if (notExists(path)) {
      throw new MojoFailureException(format("Should not have reach this point, could not obtain descriptor file from [%s]",
                                            path));
    }
    return path;
  }

  @Override
  protected void addToArchiver(ModuleArchiver archiver) throws MojoFailureException {
    super.addToArchiver(archiver);
    addArtifact(archiver, outputDirectory.toPath().resolve(finalName + ".jar").toFile());
    addDescriptor(archiver);
  }

  private void addDescriptor(ModuleArchiver archiver) throws MojoFailureException {
    final Path path = descriptorPathOrFail(outputDirectory);
    archiver.addFile(path.toFile(), MULE_PLUGIN_JSON_JAR_DESTINATION);
  }

  @Override
  protected File getMuleArtifactFile() {
    return new File(this.outputDirectory, this.finalName + "-" + MULE_SERVICE + ".jar");
  }

  @Override
  protected String getArtifactClassifier() {
    return MULE_SERVICE;
  }
}
