package com.mulesoft.adapter.helper;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.resource.ResourceException;

import org.mule.util.StringUtils;

import com.sap.aii.af.service.cpa.Channel;
import com.sap.engine.frame.core.locking.LockException;
import com.sap.engine.frame.core.locking.TechnicalLockException;
import com.sap.engine.services.applocking.LogicalLocking;
import com.sap.engine.services.applocking.LogicalLockingFactory;
import com.sap.tc.logging.Location;

/**
 * Access class to Web AS Logical locking functions
 */
public class TopicListnenerLocking {

    private static final Location LOCATION = Location.getLocation(TopicListnenerLocking.class);
    private static final String LOCK_NAMESPACE = "Mule";
    private static final String LOCK_NAME = LOCK_NAMESPACE + "/SalesforceTopic";
    private static final String LOCK_DESCRIPTION = "Mule Adapter lock";

    private final Channel channel;

    public TopicListnenerLocking(Channel channel) {
        this.channel = channel;
    }
    
    /**
     * @return null if the lock can not be aquired, TopicListnenerLocking instance otherwise
     */
    public boolean aquireLock() throws ResourceException {
        final String SIGNATURE = "aquireLock()";
        LOCATION.entering(SIGNATURE);

        boolean ret;
        
        LogicalLocking logicalLocking = getLocking();

        String argument = extractLockArgument();
        try {
            logicalLocking.lock(LogicalLocking.LIFETIME_USERSESSION, LOCK_NAME, argument, LogicalLocking.MODE_EXCLUSIVE_NONCUMULATIVE, 10000);

            ret = true;
        } catch (LockException le) {
            // the argument is already locked
            LOCATION.catching(SIGNATURE, le);

            ret = false;
        } catch (TechnicalLockException e) {
            LOCATION.catching(SIGNATURE, e);
            throw new UnsupportedOperationException("Cannot create logical lock " + LOCK_NAME + " on argument " + argument + ": " + e);
        } catch (IllegalArgumentException e) {
            LOCATION.catching(SIGNATURE, e);
            throw new UnsupportedOperationException("Cannot create logical lock " + LOCK_NAME + " on argument " + argument + ": " + e);
        }

        LOCATION.exiting(SIGNATURE, ret);
        return ret;
    }

    public void unlock() {
        final String SIGNATURE = " unlock()";
        LOCATION.entering(SIGNATURE);

        String argument = extractLockArgument();
        LogicalLocking logicalLocking = getLocking();

        try {
            logicalLocking.unlock(LogicalLocking.LIFETIME_USERSESSION, LOCK_NAME, argument, LogicalLocking.MODE_EXCLUSIVE_NONCUMULATIVE, false);
        } catch (TechnicalLockException e) {
            LOCATION.catching(SIGNATURE, e);
        } catch (IllegalArgumentException e) {
            LOCATION.catching(SIGNATURE, e);
        }

        LOCATION.exiting(SIGNATURE);
    }

    private String extractLockArgument() {
        return StringUtils.abbreviate(channel.getObjectId(), LogicalLocking.MAX_ARGUMENT_LENGTH);
    }

    private static LogicalLocking getLocking() {
        String SIGNATURE = "getLocking()";

        ClassLoader oldClassLoader = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(TopicListnenerLocking.class.getClassLoader());
        final String lookupName = "applocking";
        try {
            LogicalLockingFactory factory = (LogicalLockingFactory) new InitialContext().lookup(LogicalLockingFactory.JNDI_NAME);
            LogicalLocking logicalLocking = factory.createLogicalLocking(LOCK_NAMESPACE, LOCK_DESCRIPTION);
            return logicalLocking;
        } catch (NamingException e) {
            LOCATION.catching(SIGNATURE, e);
            throw new RuntimeException("Cannot find " + lookupName + " in JNDI");
        } catch (TechnicalLockException tle) {
            LOCATION.catching(SIGNATURE, tle);
            // the namespace is already reserved for another description
            throw new UnsupportedOperationException("Cannot set up logical locking: " + tle);
        } catch (IllegalArgumentException iae) {
            LOCATION.catching(SIGNATURE, iae);
            // the namespace is already reserved for another description
            throw new UnsupportedOperationException("Cannot set up logical locking:" + iae);
        } finally {
            Thread.currentThread().setContextClassLoader(oldClassLoader);
        }

    }

}
