/* (c) 2011-2012 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.adapter.helper;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * Helper methods for XML document.
 */
public final class XML {

    private static final String NAMESPACE_DELIMITER = ":";

    private XML() {
    }

    /**
     * @param document
     * @return all first level text nodes
     */
    public static List<List<String>> textNodes(final Document document) {
        if (document == null) {
            throw new IllegalArgumentException("null document");
        }

        final NodeList rootNodes = document.getChildNodes();
        int rootNodesLength = rootNodes.getLength();
        final List<List<String>> allTextNodes = new ArrayList<List<String>>(rootNodesLength);
        for (int i = 0; i < rootNodesLength; i++) {
            final Node rootNode = rootNodes.item(i);
            if (Node.ELEMENT_NODE == rootNode.getNodeType()) {
                NodeList childNodes = rootNode.getChildNodes();
                int childNodesLength = childNodes.getLength();
                final List<String> textNodes = new ArrayList<String>(childNodesLength);
                for (int j = 0; j < childNodesLength; j++) {
                    final Node node = childNodes.item(j);
                    if (Node.ELEMENT_NODE == node.getNodeType()) {
                        textNodes.add(node.getTextContent());
                    }
                }
                if (!textNodes.isEmpty()) {
                    allTextNodes.add(textNodes);
                }
            }
        }
        return allTextNodes;
    }

    /**
     * @param document
     * @param tagName
     * @return text node value of first occurrence of node whose tag name is
     *         `tagName`
     */
    public static String firstNamedTextNode(final Document document, final String tagName) {
        if (document == null) {
            throw new IllegalArgumentException("null document");
        }
        if (tagName == null) {
            throw new IllegalArgumentException("null tagName");
        }

        final Element rootElement = document.getDocumentElement();
        NodeList childNodes = rootElement.getChildNodes();
        int childNodesLength = childNodes.getLength();
        for (int j = 0; j < childNodesLength; j++) {
            final Node node = childNodes.item(j);
            String nodeNameWithoutNamespace = XML.stripNamespaceIfNecessary(node.getNodeName());
            if (tagName.equals(nodeNameWithoutNamespace)) {
                return node.getTextContent();
            }
        }
        return null;
    }

    /**
     * e.g. For document <tag><subtag/></tag> returns `tag`
     * 
     * @param document
     * @return root tag name
     */
    public static String rootTag(final Document document) {
        if (document == null) {
            throw new IllegalArgumentException("null document");
        }
        final NodeList nodes = document.getChildNodes();
        if (!(nodes.getLength() > 0)) {
            throw new IllegalArgumentException("Document should have at list one child node.");
        }

        return nodes.item(0).getNodeName();
    }

    /**
     * @param content
     * @return specified String as an XML {@link Document}
     * @throws ParserConfigurationException
     * @throws SAXException
     * @throws IOException
     */
    public static Document parse(final InputStream contentStream) throws ParserConfigurationException, SAXException, IOException {
        if (contentStream == null) {
            throw new IllegalArgumentException("null content");
        }

        final DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
        final DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
        return documentBuilder.parse(contentStream);
    }

    /**
     * Strip namespace of a tag if any. e.g. for `ns:tag` returns `tag`. for
     * `tag` returns `tag`.
     * 
     * @param tagName
     * @return
     */
    public static String stripNamespaceIfNecessary(final String tagName) {
        if (tagName.contains(XML.NAMESPACE_DELIMITER)) {
            return tagName.substring(tagName.indexOf(XML.NAMESPACE_DELIMITER) + 1);
        }
        return tagName;
    }

    public static byte[] emptyDocument() {
        return "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\r\n<empty/>".getBytes();
    }

}