/* (c) 2011-2012 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.adapter.module;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.resource.ResourceException;
import javax.resource.cci.InteractionSpec;
import javax.resource.cci.Record;
import javax.xml.parsers.ParserConfigurationException;

import org.mule.api.MuleContext;
import org.mule.api.MuleException;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.api.retry.RetryPolicy;
import org.mule.retry.policies.AbstractPolicyTemplate;
import org.mule.tools.module.invocation.DynamicModule;
import org.mule.tools.module.invocation.RetryingDynamicModule;
import org.mule.tools.module.model.Module;
import org.mule.tools.module.model.Module.Source;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import com.mulesoft.adapter.helper.Channels;
import com.mulesoft.adapter.helper.ExceptionHelper;
import com.mulesoft.adapter.helper.IPILogger;
import com.mulesoft.adapter.helper.PILogger;
import com.mulesoft.adapter.helper.XML;
import com.mulesoft.adapter.ra.XIMessageFactoryImpl;
import com.mulesoft.adapter.ra.XIMessageRecordImpl;
import com.sap.aii.af.lib.mp.processor.ModuleProcessor;
import com.sap.aii.af.lib.mp.processor.ModuleProcessorException;
import com.sap.aii.af.lib.mp.processor.ModuleProcessorFactory;
import com.sap.aii.af.lib.ra.cci.XIInteractionSpec;
import com.sap.aii.af.service.administration.api.monitoring.ProcessState;
import com.sap.aii.af.service.cpa.Channel;
import com.sap.engine.interfaces.messaging.api.Message;
import com.sap.engine.interfaces.messaging.api.Payload;
import com.sap.engine.interfaces.messaging.api.exception.InvalidParamException;
import com.sap.engine.interfaces.messaging.api.exception.PayloadFormatException;
import com.sap.tc.logging.Location;

/**
 * Base {@link PIModule} implementation relying on {@link DynamicModule}. <br>
 * Each interface type (aka root tag name of the xml payload) is used as
 * discriminant to dispatch to a dedicated {@link OperationHandler}.
 */
public abstract class AbstractPIModule implements PIModule {

    protected static final Location LOCATION = Location.getLocation(AbstractPIModule.class.getPackage().getName());

    private final Channel channel;
    private final DynamicModule module;
    private final Map<String, OperationHandler> operationHandlers = new ConcurrentHashMap<String, OperationHandler>();

    public AbstractPIModule(final Channel channel) throws ResourceException {
        if (channel == null) {
            throw new IllegalArgumentException("null channel");
        }

        this.channel = channel;

        final ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(MuleContext.class.getClassLoader());
        try {
            // Switch to correct ClassLoader to benefit from our own
            // dependencies.
            this.module = createConfiguredDynamicModule(channel);
        } finally {
            Thread.currentThread().setContextClassLoader(classLoader);
        }
    }

    protected final Channel getChannel() {
        return this.channel;
    }

    protected final DynamicModule getModule() {
        return this.module;
    }

    /**
     * @param channel
     * @return complete {@link DynamicModule} configured using {@link Channel}
     *         details
     * @throws ResourceException
     */
    protected final DynamicModule createConfiguredDynamicModule(final Channel channel) throws ResourceException {
        final AbstractPolicyTemplate retryPolicyTemplate = createRetryPolicyTemplate();
        IPILogger retryLogger = new PILogger(channel);
        retryPolicyTemplate.setNotifier(new DefaultRetryNotifier(retryLogger));
        return createDynamicModule(createModule(channel), createParameters(channel), retryPolicyTemplate);
    }

    /**
     * @param module
     * @param parameters
     * @param retryPolicyTemplate
     * @return
     * @throws ResourceException
     */
    protected DynamicModule createDynamicModule(final Module module, final Map<String, Object> parameters, final AbstractPolicyTemplate retryPolicyTemplate)
            throws ResourceException {
        return createDefaultDynamicModule(module, parameters, retryPolicyTemplate);
    }

    /**
     * @param module
     * @param parameters
     * @param retryPolicyTemplate
     * @return
     * @throws ResourceException
     */
    protected final DynamicModule createDefaultDynamicModule(final Module module, final Map<String, Object> parameters,
            final AbstractPolicyTemplate retryPolicyTemplate) throws ResourceException {
        return new RetryingDynamicModule(module, parameters, retryPolicyTemplate);
    }

    /**
     * @param channel
     * @return complete {@link Module} configured using {@link Channel} details
     * @throws ResourceException
     */
    protected abstract Module createModule(final Channel channel) throws ResourceException;

    /**
     * @param channel
     * @return all parameters configuring the {@link Module}
     * @throws ResourceException
     */
    protected Map<String, Object> createParameters(final Channel channel) throws ResourceException {
        return createParametersDefault(channel);
    }

    /**
     * Extract mandatory parameters from Channel by relying on
     * {@link Module#getParameters()}.
     * 
     * @param channel
     * @return
     */
    protected Map<String, Object> createParametersDefault(final Channel channel) {
        // TODO
        return Collections.emptyMap();
    }

    /**
     * @return {@link AbstractPolicyTemplate} used by
     *         {@link AbstractPIModule#createDynamicModule(com.sap.aii.af.service.cpa.Channel)}
     * @see http://www.mulesoft.org/common-retry-policies
     * @see http://www.mulesoft.org/documentation/display/MULE3USER/Configuring+
     *      Reconnection+Strategies
     */
    protected AbstractPolicyTemplate createRetryPolicyTemplate() {
        return new AbstractPolicyTemplate() {
            
            @Override
            public RetryPolicy createRetryInstance() {
                IPILogger piLogger = new PILogger(channel);
                return new DefaultRetryPolicy(AbstractPIModule.this, piLogger);
            }
        };
    }

    /**
     * @param message
     * @return operation (aka interface) encapsulated in specified
     *         {@link Message}
     * @throws ResourceException
     */
    protected final OperationHandler getOperationHandler(final Message message) throws ResourceException {
        String actionName = message.getAction().getName();
        OperationHandler matchingHandler;

        matchingHandler = operationHandlers.get(actionName);
        if (matchingHandler == null) {
            //TODO move this to SF Module
            //find the handler for an operation with an extended name like createAccount
            String lowerCaseActionName = actionName.toLowerCase();

            if (lowerCaseActionName.startsWith("create"))
                return operationHandlers.get("create");
            else if (lowerCaseActionName.startsWith("upsert"))
                return operationHandlers.get("upsert");
            else if (lowerCaseActionName.startsWith("delete"))
                return operationHandlers.get("delete");
            else if (lowerCaseActionName.startsWith("query"))
                return operationHandlers.get("query");
            else
                throw new ResourceException("Unsupported operation " + actionName + "; operation names need to start with one of "
                        + this.operationHandlers.keySet());
        }

        return matchingHandler;
    }

    @Override
    public final Record execute(final InteractionSpec spec, final XIMessageRecordImpl input, IPILogger logger) throws ResourceException {
        final String signature = "execute(InteractionSpec ispec, Record input)";
        AbstractPIModule.LOCATION.entering(signature, new Object[] { spec, input });

        if (spec == null) {
            final ResourceException exception = new ResourceException("Input ispec is null.");
            AbstractPIModule.LOCATION.throwing(signature, exception);
            throw exception;
        }
        final XIInteractionSpec interactionSpec = (XIInteractionSpec) spec;

        // Switch to correct ClassLoader during operation execution.
        final ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(MuleContext.class.getClassLoader());
        try {
            final Message message = input.getXIMessage();
            // Dispatch to correct OperationHandler.
            final byte[] payload = dispatch(message, logger);
            final String functionName = interactionSpec.getFunctionName();
            // Based on XIInteractionSpec, return result as a Record or not.
            if (XIInteractionSpec.SEND.equals(functionName)) {
                if (payload != null) {
                    AbstractPIModule.LOCATION.warningT("Function name is " + XIInteractionSpec.SEND + " but an output has been generated: <{0}>",
                            new Object[] { payload });
                    logger.reportProcessingStatus(ProcessState.ERROR, "The operation is called asynchronous but has generated a response from Salesforce");
                }

                return null;
            } else if (XIInteractionSpec.CALL.equals(functionName)) {
                if (payload == null) {
                    AbstractPIModule.LOCATION.warningT("Function name is " + XIInteractionSpec.CALL + " but no output has been generated");
                    logger.reportProcessingStatus(ProcessState.ERROR, "The operation is called synchronous but has not generated a response from Salesforce");
                }

                return createOutput(message, payload, logger);
            } else {
                throw new IllegalArgumentException("Unsupported function name <" + functionName + ">");
            }
        } catch (ResourceException e) {
            throw e;
        }  catch (Exception e) {
            String rootCauseMessage = ExceptionHelper.extractRootCauseMessage(e);
            throw new ResourceException(e.getMessage() + ", root cause is: " + rootCauseMessage, e);
        } finally {
            Thread.currentThread().setContextClassLoader(classLoader);

            AbstractPIModule.LOCATION.exiting(signature);
        }
    }
    
    public boolean retryOn(Throwable cause) {
        return false;
    }
    
    /**
     * Register an {@link OperationHandler} with this {@link PIModule}. Will be
     * considered during
     * {@link PIModule#execute(javax.resource.cci.InteractionSpec, javax.resource.cci.Record)}
     * .
     * 
     * @param handler
     */
    protected final void register(final OperationHandler handler) {
        this.operationHandlers.put(handler.getOperationName(), handler);
    }

    /**
     * @param message
     * @param payload
     * @param logger 
     * @return a {@link Record} encapsulating specified `payload`. Useful as an
     *         PI output.
     * @throws ResourceException
     * @throws IOException
     * @throws InvalidParamException
     * @throws PayloadFormatException
     * @see #execute(javax.resource.cci.InteractionSpec,
     *      javax.resource.cci.Record)
     */
    protected final Record createOutput(final Message message, final byte[] payload, IPILogger logger) throws ResourceException, IOException, InvalidParamException,
            PayloadFormatException {
        final XIMessageRecordImpl output = new XIMessageRecordImpl(message.getToParty(), message.getFromParty(), message.getToService(),
                message.getFromService(), message.getAction());
        final Message responseMessage = output.getXIMessage();
        
        responseMessage.setRefToMessageId(message.getMessageId());
        logger.setPiMessage(responseMessage);

        if (payload == null) {
            byte[] dummyDocument = XML.emptyDocument();
            logger.reportProcessingStatus(ProcessState.OK, "Sending emty response message", message);
            com.mulesoft.adapter.helper.Payload.populateMessage(responseMessage, dummyDocument);
        } else {
            logger.reportProcessingStatus(ProcessState.OK, "Sending response message", message);
            com.mulesoft.adapter.helper.Payload.populateMessage(responseMessage, payload);
        }
        return output;
    }

    /**
     * Dispatch execution to the correct {@link OperationHandler}.
     * 
     * @param message
     * @return
     * @throws ResourceException
     * @throws InitialisationException
     * @throws MuleException
     * @see #extractPayload(javax.resource.cci.Record)
     * @see #extractOperation(javax.resource.cci.Record)
     */
    protected byte[] dispatch(final Message message, IPILogger logger) throws ResourceException, MuleException, IOException, InvalidParamException,
            PayloadFormatException, ParserConfigurationException, SAXException {
        final OperationHandler operationHandler = getOperationHandler(message);

        Payload mainPayload = message.getMainPayload();
        InputStream mainPayloadInputStream = mainPayload.getInputStream();
        final Document document = XML.parse(mainPayloadInputStream);
        AbstractPIModule.LOCATION.debugT("Received XML payload");
        
        String responseRootElementName = Channels.retrieveQueryResponseRootElementName(channel);
        String responseRootElementNamespace = Channels.retrieveResponseRootElementNamespace(channel);
        PIMessageParameters messageParameters = new PIMessageParameters(document, responseRootElementName, responseRootElementNamespace, logger);

        return operationHandler.handle(messageParameters);
    }

    /**
     * Subscribe to a named {@link Source} associated to specified
     * {@link Channel}.
     * 
     * @param sourceName
     * @param overridenParameters
     * @param channel
     * @param messageFactory
     * @throws InitialisationException
     * @throws MuleException
     * @throws ModuleProcessorException
     */
    protected final void subscribe(final String sourceName, final Map<String, Object> overridenParameters, final XIMessageFactoryImpl messageFactory)
            throws MuleException, ModuleProcessorException {
        final ModuleProcessor moduleProcessor = ModuleProcessorFactory.getModuleProcessor(true, 1, 1000);
        getModule().subscribe(sourceName, overridenParameters, new SubscriptionEventListener(messageFactory, this.channel, moduleProcessor));
    }

    /**
     * Unsubscribe from a named {@link Source}.
     * 
     * @param sourceName
     * @throws MuleException
     */
    protected final void unsubscribe(final String sourceName) throws MuleException {
        getModule().unsubscribe(sourceName);
    }

    @Override
    public void dispose() {
        disposeDefault();
    }

    /**
     * Dispose underlying {@link DynamicModule}.
     * 
     * @see DynamicModule#dispose()
     */
    protected final void disposeDefault() {
        this.module.dispose();
    }

}