package com.mulesoft.adapter.module;

import java.util.Map;

import javax.resource.ResourceException;

import org.mule.api.MuleMessage;
import org.mule.tools.module.invocation.DynamicModule;
import org.mule.util.StringUtils;

import com.mulesoft.adapter.helper.Channels;
import com.mulesoft.adapter.helper.ExceptionHelper;
import com.mulesoft.adapter.helper.IPILogger;
import com.mulesoft.adapter.helper.PILogger;
import com.mulesoft.adapter.helper.Payload;
import com.mulesoft.adapter.module.salesforce.SObjects;
import com.mulesoft.adapter.ra.XIMessageFactoryImpl;
import com.sap.aii.af.lib.mp.module.ModuleData;
import com.sap.aii.af.lib.mp.processor.ModuleProcessor;
import com.sap.aii.af.service.administration.api.cpa.CPAChannelStoppedException;
import com.sap.aii.af.service.administration.api.cpa.CPAFactory;
import com.sap.aii.af.service.administration.api.monitoring.ProcessState;
import com.sap.aii.af.service.cpa.Binding;
import com.sap.aii.af.service.cpa.CPAException;
import com.sap.aii.af.service.cpa.Channel;
import com.sap.aii.af.service.util.transaction.api.TxManager;
import com.sap.aii.af.service.util.transaction.api.TxManagerException;
import com.sap.aii.af.service.util.transaction.api.TxManagerFactory;
import com.sap.aii.af.service.util.transaction.api.TxTicket;
import com.sap.engine.interfaces.messaging.api.DeliverySemantics;
import com.sap.engine.interfaces.messaging.api.Message;
import com.sap.engine.interfaces.messaging.api.auditlog.AuditLogStatus;
import com.sap.engine.interfaces.messaging.api.exception.InvalidParamException;
import com.sap.engine.interfaces.messaging.api.exception.PayloadFormatException;
import com.sap.tc.logging.Location;

public class SubscriptionEventListener implements DynamicModule.Listener {

    private static final Location LOCATION = Location.getLocation(SubscriptionEventListener.class);

    private final ModuleProcessor moduleProcessor;
    private final Channel channel;
    private final XIMessageFactoryImpl messageFactory;

    public SubscriptionEventListener(final XIMessageFactoryImpl messageFactory, final Channel channel, final ModuleProcessor moduleProcessor) {
        this.messageFactory = messageFactory;
        this.channel = channel;
        this.moduleProcessor = moduleProcessor;
    }

    @Override
    public void onEvent(final MuleMessage event) {
        final String signature = "onEvent(final Map<String, Object> event)";
        SubscriptionEventListener.LOCATION.entering(signature, new Object[] { event });

        final IPILogger logger = new PILogger(this.channel);
        try {
            final String channelID = this.channel.getObjectId();
            final Message message = createPIMessage(channelID);

            logger.setPiMessage(message);
            logger.reportProcessingStatus(ProcessState.OK, "Event received");
            logger.reportAuditStatus(AuditLogStatus.SUCCESS, "Message created from Salesfoce topic event");

            setDeliverySemantics(message);
            final byte[] payload = generatePayload(event, logger);
            Payload.populateMessage(message, payload);

            final ModuleData moduleData = new ModuleData();
            moduleData.setPrincipalData(message);

            sendMessageToModuleProcessor(channelID, moduleData, logger);
        } catch (RuntimeException e) {
            SubscriptionEventListener.LOCATION.catching(signature, e);
            logger.reportProcessingStatus(ProcessState.FATAL, "Error when receiving subscribed event: {0}", e.getMessage());
        } catch (CPAChannelStoppedException e) {
            SubscriptionEventListener.LOCATION.catching(signature, e);
            logger.reportProcessingStatus(ProcessState.FATAL, "Error when delivering message from subscribed event: {0}", e.getMessage());
        } catch (CPAException e) {
            SubscriptionEventListener.LOCATION.catching(signature, e);
            logger.reportProcessingStatus(ProcessState.FATAL, "Can not determine binding for channel for subscribed event: {0}", e.getMessage());
        } catch (ResourceException e) {
            SubscriptionEventListener.LOCATION.catching(signature, e);
            logger.reportProcessingStatus(ProcessState.FATAL, "Can not create message for subscribed event: {0}", e.getMessage());
        } catch (InvalidParamException e) {
            SubscriptionEventListener.LOCATION.catching(signature, e);
            logger.reportProcessingStatus(ProcessState.FATAL, "Error when receiving subscribed event: {0}", e.getMessage());
        } catch (PayloadFormatException e) {
            SubscriptionEventListener.LOCATION.catching(signature, e);
            logger.reportProcessingStatus(ProcessState.FATAL, "Error when receiving subscribed event: {0}", e.getMessage());
        }

        SubscriptionEventListener.LOCATION.exiting(signature);
    }

    protected final byte[] generatePayload(final MuleMessage message, IPILogger logger) throws ResourceException {
        final String rootElementName = Channels.retrieveSubscriptionResponseRootElementName(channel);
        final String rootElementNamespace = Channels.retrieveResponseRootElementNamespace(channel);
        
        final Map<String, Object> messagePayload = (Map<String, Object>) message.getPayload();
        final String inboundChannel = message.getInboundProperty("channel");
        final String createdDate = message.getInboundProperty("createdDate");
        final String type = message.getInboundProperty("type");

        logger.reportAuditStatus(AuditLogStatus.SUCCESS, "event for type {0} created at {1} received via channel {2}", type, createdDate, inboundChannel);

        return SObjects.generateSubscriptionResponse(messagePayload, rootElementName, rootElementNamespace, inboundChannel, createdDate, type);
    }

    private void sendMessageToModuleProcessor(final String channelID, final ModuleData moduleData, IPILogger logger) throws ResourceException {
        final String signature = "sendMessageToModuleProcessor(final String channelID, final ModuleData moduleData)";
        SubscriptionEventListener.LOCATION.entering(signature, new Object[] { channelID, moduleData });

        TxManager txManager = null;
        TxTicket tx = null;
        try {
            txManager = TxManagerFactory.getInstance().getTxManager();
            tx = txManager.beginTransaction();

            this.moduleProcessor.process(channelID, moduleData);
            
            txManager.commit(tx);
            
        } catch (TxManagerException e) {
            SubscriptionEventListener.LOCATION.catching(signature, e);
            
            throw new ResourceException("Cannot commit transaction: " + e, e);
        } catch (Exception e) {
            SubscriptionEventListener.LOCATION.catching(signature, e);

            if (tx != null && txManager != null) {
                try {
                    txManager.rollback(tx);
                } catch (TxManagerException e1) {
                    SubscriptionEventListener.LOCATION.catching(signature, e1);
                    logger.reportProcessingStatus(ProcessState.FATAL, "Cannot roll back transaction: {0}", e1);
                    throw new ResourceException("Cannot roll back transaction: " + e1, e1);
                }
            }

            logger.reportProcessingStatus(ProcessState.FATAL, "Cannot send message to module processor: {0}, root cause: {1}", e, ExceptionHelper.extractRootCauseMessage(e));
            throw new ResourceException("Cannot send message to module processor " + e, e);
        }

    }

    protected final Message createPIMessage(final String channelID) throws CPAException, ResourceException {
        final Binding binding = CPAFactory.getInstance().getLookupManager().getBindingByChannelId(channelID);
        final Message message = this.messageFactory.createMessageRecord(binding.getFromParty(), binding.getToParty(), binding.getFromService(), binding.getToService(), binding.getActionName(), binding.getActionNamespace());
        return message;
    }

    protected final void setDeliverySemantics(final Message message) throws InvalidParamException, ResourceException {
        final DeliverySemantics qualityOfService = Channels.retrieveQualityOfService(this.channel);
        message.setDeliverySemantics(DeliverySemantics.ExactlyOnce);

        if (qualityOfService == DeliverySemantics.ExactlyOnceInOrder) {
            final String queueName = Channels.retrieveQueueName(channel);
            message.setSequenceId(StringUtils.deleteWhitespace(queueName));
        }
    }

}
