/* (c) 2011-2012 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.adapter.module.salesforce;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.resource.ResourceException;

import org.mule.api.MuleException;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.tools.module.invocation.DynamicModule;

import com.sforce.async.BatchInfo;
import com.sforce.async.BatchResult;
import com.sforce.async.Result;

/**
 * Helper methods for bulk operations.
 */
public final class Bulks {

    private static final long CHECK_INTERVAL = 1000L;
    private static final String BATCH_INFO_PROCESSOR_NAME = "batch-info";
    private static final String BATCH_RESULT_PROCESSOR_NAME = "batch-result";
    private static final String BATCH_INFO_PARAMETER_NAME = "batchInfo";

    private Bulks() {
    }

    /**
     * @param module
     * @param batchInfo
     * @return latest {@link BatchInfo}
     * @throws InitialisationException
     * @throws MuleException 
     */
    public static BatchInfo batchInfo(final DynamicModule module, final BatchInfo batchInfo) throws MuleException {
        final Map<String, Object> parameters = new HashMap<String, Object>();
        parameters.put(Bulks.BATCH_INFO_PARAMETER_NAME, batchInfo);
        return module.invoke(Bulks.BATCH_INFO_PROCESSOR_NAME, parameters);
    }

    /**
     * Await completion of specified {@link BatchInfo}. Loop calling {@link #batchInfo(org.mule.tools.module.invocation.DynamicModule, com.sforce.async.BatchInfo)} waiting 10 s between each call.
     * @param module
     * @param batchInfo
     * @return
     * @throws InitialisationException
     * @throws MuleException
     * @throws InterruptedException
     * @see #batchInfo(org.mule.tools.module.invocation.DynamicModule, com.sforce.async.BatchInfo) 
     */
    public static BatchResult awaitCompletion(final DynamicModule module, final BatchInfo batchInfo) throws MuleException, InterruptedException {
        while (true) {
            final BatchInfo latestBatchInfo = Bulks.batchInfo(module, batchInfo);
            switch (latestBatchInfo.getState()) {
                case Completed:
                case Failed:
                    final Map<String, Object> parameters = new HashMap<String, Object>();
                    parameters.put(Bulks.BATCH_INFO_PARAMETER_NAME, batchInfo);
                    return module.invoke(Bulks.BATCH_RESULT_PROCESSOR_NAME, parameters);
            }

            Thread.sleep(Bulks.CHECK_INTERVAL);
        }
    }

    /**
     * Throws a {@link ResourceException} if specified {@link BatchResult} contains {@link com.sforce.async.Error}.
     * @param batchResult
     * @throws ResourceException 
     */
    public static void throwIfNotSuccessful(final BatchResult batchResult) throws ResourceException {
        final List<Result> failures = new LinkedList<Result>();
        for (final Result result : batchResult.getResult()) {
            if (!result.isSuccess()) {
                failures.add(result);
            }
        }
        if (!failures.isEmpty()) {
            final StringBuilder builder = new StringBuilder();
            for (final Result failure : failures) {
                builder.append(failure.getId()).append(" {\n");
                for (final com.sforce.async.Error error : failure.getErrors()) {
                    builder.append(error.getStatusCode()).append(":").append(error.getMessage()).append(" ");
                }
                builder.append("}\n");
            }
            throw new ResourceException(builder.toString());
        }
    }

}